/* gethdr.c, header decoding                                                */

/* Copyright (C) 1996, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include "config.h"

#include <stdio.h>

#include "mpeg-config.h"
#include "global.h"


/* private prototypes */
static void sequence_header _ANSI_ARGS_((struct mpeg2obj *m));
static void group_of_pictures_header _ANSI_ARGS_((struct mpeg2obj *m));
static void picture_header _ANSI_ARGS_((struct mpeg2obj *m));
static void extension_and_user_data _ANSI_ARGS_((struct mpeg2obj *m));
static void sequence_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void sequence_display_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void quant_matrix_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void sequence_scalable_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void picture_display_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void picture_coding_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void picture_spatial_scalable_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void picture_temporal_scalable_extension _ANSI_ARGS_((struct mpeg2obj *m));
static int  extra_bit_information _ANSI_ARGS_((struct mpeg2obj *m));
static void copyright_extension _ANSI_ARGS_((struct mpeg2obj *m));
static void user_data _ANSI_ARGS_((struct mpeg2obj *m));
static void user_data _ANSI_ARGS_((struct mpeg2obj *m));




/* introduced in September 1995 to assist spatial scalable decoding */
static void Update_Temporal_Reference_Tacking_Data _ANSI_ARGS_((struct mpeg2obj *m));
/* private variables */

#define RESERVED    -1 
static double frame_rate_Table[16] =
{
  0.0,
  ((23.0*1000.0)/1001.0),
  24.0,
  25.0,
  ((30.0*1000.0)/1001.0),
  30.0,
  50.0,
  ((60.0*1000.0)/1001.0),
  60.0,
 
  RESERVED,
  RESERVED,
  RESERVED,
  RESERVED,
  RESERVED,
  RESERVED,
  RESERVED
};

/*
 * decode headers from one input stream
 * until an End of Sequence or picture start code
 * is found
 */
int _mpeg2Get_Hdr(struct mpeg2obj *m)
{
  unsigned int code;

  for (;;)
  {
    /* look for next_start_code */
    _mpeg2next_start_code(m);
    code = _mpeg2Get_Bits32(m);
  
    switch (code)
    {
    case SEQUENCE_HEADER_CODE:
      sequence_header(m);
      break;
    case GROUP_START_CODE:
      group_of_pictures_header(m);
      break;
    case PICTURE_START_CODE:
      picture_header(m);
      return 1;
      break;
    case SEQUENCE_END_CODE:
      return 0;
      break;
    default:
      if (!_mpeg2Quiet_Flag)
        fprintf(stderr,"Unexpected next_start_code %08x (ignored)\n",code);
      break;
    }
  }
}


/* align to start of next next_start_code */

void _mpeg2next_start_code(struct mpeg2obj *m)
{
  /* byte align */
  _mpeg2Flush_Buffer(m,m->ld->Incnt&7);
  while (_mpeg2Show_Bits(m,24)!=0x01L)
    _mpeg2Flush_Buffer(m,8);
}


/* decode sequence header */

static void sequence_header(struct mpeg2obj *m)
{
  int i;
  int pos;

  pos = m->ld->Bitcnt;
  m->horizontal_size             = _mpeg2Get_Bits(m,12);
  m->vertical_size               = _mpeg2Get_Bits(m,12);
  m->aspect_ratio_information    = _mpeg2Get_Bits(m,4);
  m->frame_rate_code             = _mpeg2Get_Bits(m,4);
  m->bit_rate_value              = _mpeg2Get_Bits(m,18);
  _mpeg2marker_bit(m,"sequence_header()");
  m->vbv_buffer_size             = _mpeg2Get_Bits(m,10);
  m->constrained_parameters_flag = _mpeg2Get_Bits(m,1);

  if((m->ld->load_intra_quantizer_matrix = _mpeg2Get_Bits(m,1)))
  {
    for (i=0; i<64; i++)
      m->ld->intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]] = _mpeg2Get_Bits(m,8);
  }
  else
  {
    for (i=0; i<64; i++)
      m->ld->intra_quantizer_matrix[i] = _mpeg2default_intra_quantizer_matrix[i];
  }

  if((m->ld->load_non_intra_quantizer_matrix = _mpeg2Get_Bits(m,1)))
  {
    for (i=0; i<64; i++)
      m->ld->non_intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]] = _mpeg2Get_Bits(m,8);
  }
  else
  {
    for (i=0; i<64; i++)
      m->ld->non_intra_quantizer_matrix[i] = 16;
  }

  /* copy luminance to chrominance matrices */
  for (i=0; i<64; i++)
  {
    m->ld->chroma_intra_quantizer_matrix[i] =
      m->ld->intra_quantizer_matrix[i];

    m->ld->chroma_non_intra_quantizer_matrix[i] =
      m->ld->non_intra_quantizer_matrix[i];
  }

#ifdef VERBOSE
  if (Verbose_Flag > NO_LAYER)
  {
    printf("sequence header (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag > SEQUENCE_LAYER)
    {
      printf("  horizontal_size=%d\n",horizontal_size);
      printf("  vertical_size=%d\n",vertical_size);
      printf("  aspect_ratio_information=%d\n",aspect_ratio_information);
      printf("  frame_rate_code=%d",frame_rate_code);
      printf("  bit_rate_value=%d\n",bit_rate_value);
      printf("  vbv_buffer_size=%d\n",vbv_buffer_size);
      printf("  constrained_parameters_flag=%d\n",constrained_parameters_flag);
      printf("  load_intra_quantizer_matrix=%d\n",ld->load_intra_quantizer_matrix);
      printf("  load_non_intra_quantizer_matrix=%d\n",ld->load_non_intra_quantizer_matrix);
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_sequence_header++;
#endif /* VERIFY */

  extension_and_user_data(m);
}



/* decode group of pictures header */
/* ISO/IEC 13818-2 section 6.2.2.6 */
static void group_of_pictures_header(struct mpeg2obj *m)
{
  int pos;

  if (m->ld == &m->base)
  {
    m->Temporal_Reference_Base = m->True_Framenum_max + 1; 	/* *CH* */
    m->Temporal_Reference_GOP_Reset = 1;
  }
  pos = m->ld->Bitcnt;
  m->drop_flag   = _mpeg2Get_Bits(m,1);
  m->hour        = _mpeg2Get_Bits(m,5);
  m->minute      = _mpeg2Get_Bits(m,6);
  _mpeg2marker_bit(m,"group_of_pictures_header()");
  m->sec         = _mpeg2Get_Bits(m,6);
  m->frame       = _mpeg2Get_Bits(m,6);
  m->closed_gop  = _mpeg2Get_Bits(m,1);
  m->broken_link = _mpeg2Get_Bits(m,1);

#ifdef VERBOSE
  if (Verbose_Flag > NO_LAYER)
  {
    printf("group of pictures (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag > SEQUENCE_LAYER)
    {
      printf("  drop_flag=%d\n",drop_flag);
      printf("  timecode %d:%02d:%02d:%02d\n",hour,minute,sec,frame);
      printf("  closed_gop=%d\n",closed_gop);
      printf("  broken_link=%d\n",broken_link);
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_group_of_pictures_header++;
#endif /* VERIFY */

  extension_and_user_data(m);

}


/* decode picture header */

/* ISO/IEC 13818-2 section 6.2.3 */
static void picture_header(struct mpeg2obj *m)
{
  int pos;
  int Extra_Information_Byte_Count;

  /* unless later overwritten by picture_spatial_scalable_extension() */
  m->ld->pict_scal = 0; 
  
  pos = m->ld->Bitcnt;
  m->temporal_reference  = _mpeg2Get_Bits(m,10);
  m->picture_coding_type = _mpeg2Get_Bits(m,3);
  m->vbv_delay           = _mpeg2Get_Bits(m,16);

  if (m->picture_coding_type==P_TYPE || m->picture_coding_type==B_TYPE)
  {
    m->full_pel_forward_vector = _mpeg2Get_Bits(m,1);
    m->forward_f_code = _mpeg2Get_Bits(m,3);
  }
  if (m->picture_coding_type==B_TYPE)
  {
    m->full_pel_backward_vector = _mpeg2Get_Bits(m,1);
    m->backward_f_code = _mpeg2Get_Bits(m,3);
  }

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("picture header (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SEQUENCE_LAYER)
    {
      printf("  temporal_reference=%d\n",temporal_reference);
      printf("  picture_coding_type=%d\n",picture_coding_type);
      printf("  vbv_delay=%d\n",vbv_delay);
      if (picture_coding_type==P_TYPE || picture_coding_type==B_TYPE)
      {
        printf("  full_pel_forward_vector=%d\n",full_pel_forward_vector);
        printf("  forward_f_code =%d\n",forward_f_code);
      }
      if (picture_coding_type==B_TYPE)
      {
        printf("  full_pel_backward_vector=%d\n",full_pel_backward_vector);
        printf("  backward_f_code =%d\n",backward_f_code);
      }
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_picture_header++;
#endif /* VERIFY */

  Extra_Information_Byte_Count = 
    extra_bit_information(m);
  
  extension_and_user_data(m);

  /* update tracking information used to assist spatial scalability */
  Update_Temporal_Reference_Tacking_Data(m);
}

/* decode slice header */

/* ISO/IEC 13818-2 section 6.2.4 */
int _mpeg2slice_header(struct mpeg2obj *m)
{
  int slice_vertical_position_extension;
  int quantizer_scale_code;
  int pos;
  int slice_picture_id_enable = 0;
  int slice_picture_id = 0;
  int extra_information_slice = 0;

  pos = m->ld->Bitcnt;

  slice_vertical_position_extension =
    (m->ld->MPEG2_Flag && m->vertical_size>2800) ? _mpeg2Get_Bits(m,3) : 0;

  if (m->ld->scalable_mode==SC_DP)
    m->ld->priority_breakpoint = _mpeg2Get_Bits(m,7);

  quantizer_scale_code = _mpeg2Get_Bits(m,5);
  m->ld->quantizer_scale =
    m->ld->MPEG2_Flag ? (m->ld->q_scale_type ? _mpeg2Non_Linear_quantizer_scale[quantizer_scale_code] : quantizer_scale_code<<1) : quantizer_scale_code;

  /* slice_id introduced in March 1995 as part of the video corridendum
     (after the IS was drafted in November 1994) */
  if (_mpeg2Get_Bits(m,1))
  {
    m->ld->intra_slice = _mpeg2Get_Bits(m,1);

    slice_picture_id_enable = _mpeg2Get_Bits(m,1);
	slice_picture_id = _mpeg2Get_Bits(m,6);

    extra_information_slice = extra_bit_information(m);
  }
  else
    m->ld->intra_slice = 0;

#ifdef VERBOSE
  if (Verbose_Flag>PICTURE_LAYER)
  {
    printf("slice header (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SLICE_LAYER)
    {
      if (ld->MPEG2_Flag && vertical_size>2800)
        printf("  slice_vertical_position_extension=%d\n",slice_vertical_position_extension);
  
      if (ld->scalable_mode==SC_DP)
        printf("  priority_breakpoint=%d\n",ld->priority_breakpoint);

      printf("  quantizer_scale_code=%d\n",quantizer_scale_code);

      printf("  slice_picture_id_enable = %d\n", slice_picture_id_enable);

      if(slice_picture_id_enable)
        printf("  slice_picture_id = %d\n", slice_picture_id);

    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_slice_header++;
#endif /* VERIFY */


  return slice_vertical_position_extension;
}


/* decode extension and user data */
/* ISO/IEC 13818-2 section 6.2.2.2 */
static void extension_and_user_data(struct mpeg2obj *m)
{
  int code,ext_ID;

  _mpeg2next_start_code(m);

  while ((code = _mpeg2Show_Bits(m,32))==EXTENSION_START_CODE || code==USER_DATA_START_CODE)
  {
    if (code==EXTENSION_START_CODE)
    {
      _mpeg2Flush_Buffer32(m);
      ext_ID = _mpeg2Get_Bits(m,4);
      switch (ext_ID)
      {
      case SEQUENCE_EXTENSION_ID:
        sequence_extension(m);
        break;
      case SEQUENCE_DISPLAY_EXTENSION_ID:
        sequence_display_extension(m);
        break;
      case QUANT_MATRIX_EXTENSION_ID:
        quant_matrix_extension(m);
        break;
      case SEQUENCE_SCALABLE_EXTENSION_ID:
        sequence_scalable_extension(m);
        break;
      case PICTURE_DISPLAY_EXTENSION_ID:
        picture_display_extension(m);
        break;
      case PICTURE_CODING_EXTENSION_ID:
        picture_coding_extension(m);
        break;
      case PICTURE_SPATIAL_SCALABLE_EXTENSION_ID:
        picture_spatial_scalable_extension(m);
        break;
      case PICTURE_TEMPORAL_SCALABLE_EXTENSION_ID:
        picture_temporal_scalable_extension(m);
        break;
      case COPYRIGHT_EXTENSION_ID:
        copyright_extension(m);
        break;
     default:
        fprintf(stderr,"reserved extension start code ID %d\n",ext_ID);
        break;
      }
      _mpeg2next_start_code(m);
    }
    else
    {
#ifdef VERBOSE
      if (Verbose_Flag>NO_LAYER)
        printf("user data\n");
#endif /* VERBOSE */
      _mpeg2Flush_Buffer32(m);
      user_data(m);
    }
  }
}


/* decode sequence extension */

/* ISO/IEC 13818-2 section 6.2.2.3 */
static void sequence_extension(struct mpeg2obj *m)
{
  int horizontal_size_extension;
  int vertical_size_extension;
  int bit_rate_extension;
  int vbv_buffer_size_extension;
  /* int pos; */

  /* derive bit position for trace */
#ifdef VERBOSE
  pos = ld->Bitcnt;
#endif

  m->ld->MPEG2_Flag = 1;

  m->ld->scalable_mode = SC_NONE; /* unless overwritten by sequence_scalable_extension() */
  m->layer_id = 0;                /* unless overwritten by sequence_scalable_extension() */
  
  m->profile_and_level_indication = _mpeg2Get_Bits(m,8);
  m->progressive_sequence         = _mpeg2Get_Bits(m,1);
  m->chroma_format                = _mpeg2Get_Bits(m,2);
  horizontal_size_extension    = _mpeg2Get_Bits(m,2);
  vertical_size_extension      = _mpeg2Get_Bits(m,2);
  bit_rate_extension           = _mpeg2Get_Bits(m,12);
  _mpeg2marker_bit(m,"sequence_extension");
  vbv_buffer_size_extension    = _mpeg2Get_Bits(m,8);
  m->low_delay                    = _mpeg2Get_Bits(m,1);
  m->frame_rate_extension_n       = _mpeg2Get_Bits(m,2);
  m->frame_rate_extension_d       = _mpeg2Get_Bits(m,5);

  m->frame_rate = frame_rate_Table[m->frame_rate_code] *
    ((m->frame_rate_extension_n+1)/(m->frame_rate_extension_d+1));

  /* special case for 422 profile & level must be made */
  if((m->profile_and_level_indication>>7) & 1)
  {  /* escape bit of profile_and_level_indication set */
  
    /* 4:2:2 Profile @ Main Level */
    if((m->profile_and_level_indication&15)==5)
    {
      m->profile = PROFILE_422;
      m->level   = MAIN_LEVEL;  
    }
  }
  else
  {
    m->profile = m->profile_and_level_indication >> 4;  /* Profile is upper nibble */
    m->level   = m->profile_and_level_indication & 0xF;  /* Level is lower nibble */
  }
  
 
  m->horizontal_size = (horizontal_size_extension<<12) | (m->horizontal_size&0x0fff);
  m->vertical_size = (vertical_size_extension<<12) | (m->vertical_size&0x0fff);


  /* ISO/IEC 13818-2 does not define bit_rate_value to be composed of
   * both the original bit_rate_value parsed in sequence_header() and
   * the optional bit_rate_extension in sequence_extension_header(). 
   * However, we use it for bitstream verification purposes. 
   */

  m->bit_rate_value += (bit_rate_extension << 18);
  m->bit_rate = ((double) m->bit_rate_value) * 400.0;
  m->vbv_buffer_size += (vbv_buffer_size_extension << 10);

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("sequence extension (byte %d)\n",(pos>>3)-4);

    if (Verbose_Flag>SEQUENCE_LAYER)
    {
      printf("  profile_and_level_indication=%d\n",profile_and_level_indication);

      if (profile_and_level_indication<128)
      {
        printf("    profile=%d, level=%d\n",profile,level);
      }

      printf("  progressive_sequence=%d\n",progressive_sequence);
      printf("  chroma_format=%d\n",chroma_format);
      printf("  horizontal_size_extension=%d\n",horizontal_size_extension);
      printf("  vertical_size_extension=%d\n",vertical_size_extension);
      printf("  bit_rate_extension=%d\n",bit_rate_extension);
      printf("  vbv_buffer_size_extension=%d\n",vbv_buffer_size_extension);
      printf("  low_delay=%d\n",low_delay);
      printf("  frame_rate_extension_n=%d\n",frame_rate_extension_n);
      printf("  frame_rate_extension_d=%d\n",frame_rate_extension_d);
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_sequence_extension++;
#endif /* VERIFY */


}


/* decode sequence display extension */

static void sequence_display_extension(struct mpeg2obj *m)
{
  int pos;

  pos = m->ld->Bitcnt;
  m->video_format      = _mpeg2Get_Bits(m,3);
  m->color_description = _mpeg2Get_Bits(m,1);

  if (m->color_description)
  {
    m->color_primaries          = _mpeg2Get_Bits(m,8);
    m->transfer_characteristics = _mpeg2Get_Bits(m,8);
    m->matrix_coefficients      = _mpeg2Get_Bits(m,8);
  }

  m->display_horizontal_size = _mpeg2Get_Bits(m,14);
  _mpeg2marker_bit(m,"sequence_display_extension");
  m->display_vertical_size   = _mpeg2Get_Bits(m,14);

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("sequence display extension (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SEQUENCE_LAYER)
    {

      printf("  video_format=%d\n",video_format);
      printf("  color_description=%d\n",color_description);

      if (color_description)
      {
        printf("    color_primaries=%d\n",color_primaries);
        printf("    transfer_characteristics=%d\n",transfer_characteristics);
        printf("    matrix_coefficients=%d\n",matrix_coefficients);
      }
      printf("  display_horizontal_size=%d\n",display_horizontal_size);
      printf("  display_vertical_size=%d\n",display_vertical_size);
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_sequence_display_extension++;
#endif /* VERIFY */

}


/* decode quant matrix entension */
/* ISO/IEC 13818-2 section 6.2.3.2 */
static void quant_matrix_extension(struct mpeg2obj *m)
{
  int i;
  int pos;

  pos = m->ld->Bitcnt;

  if((m->ld->load_intra_quantizer_matrix = _mpeg2Get_Bits(m,1)))
  {
    for (i=0; i<64; i++)
    {
      m->ld->chroma_intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]]
      = m->ld->intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]]
      = _mpeg2Get_Bits(m,8);
    }
  }

  if((m->ld->load_non_intra_quantizer_matrix = _mpeg2Get_Bits(m,1)))
  {
    for (i=0; i<64; i++)
    {
      m->ld->chroma_non_intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]]
      = m->ld->non_intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]]
      = _mpeg2Get_Bits(m,8);
    }
  }

  if((m->ld->load_chroma_intra_quantizer_matrix = _mpeg2Get_Bits(m,1)))
  {
    for (i=0; i<64; i++)
      m->ld->chroma_intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]] = _mpeg2Get_Bits(m,8);
  }

  if((m->ld->load_chroma_non_intra_quantizer_matrix = _mpeg2Get_Bits(m,1)))
  {
    for (i=0; i<64; i++)
      m->ld->chroma_non_intra_quantizer_matrix[_mpeg2scan[ZIG_ZAG][i]] = _mpeg2Get_Bits(m,8);
  }

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("quant matrix extension (byte %d)\n",(pos>>3)-4);
    printf("  load_intra_quantizer_matrix=%d\n",
      ld->load_intra_quantizer_matrix);
    printf("  load_non_intra_quantizer_matrix=%d\n",
      ld->load_non_intra_quantizer_matrix);
    printf("  load_chroma_intra_quantizer_matrix=%d\n",
      ld->load_chroma_intra_quantizer_matrix);
    printf("  load_chroma_non_intra_quantizer_matrix=%d\n",
      ld->load_chroma_non_intra_quantizer_matrix);
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_quant_matrix_extension++;
#endif /* VERIFY */

}


/* decode sequence scalable extension */
/* ISO/IEC 13818-2   section 6.2.2.5 */
static void sequence_scalable_extension(struct mpeg2obj *m)
{
  int pos;

  pos = m->ld->Bitcnt;

  /* values (without the +1 offset) of scalable_mode are defined in 
     Table 6-10 of ISO/IEC 13818-2 */
  m->ld->scalable_mode = _mpeg2Get_Bits(m,2) + 1; /* add 1 to make SC_DP != SC_NONE */

  m->layer_id = _mpeg2Get_Bits(m,4);

  if (m->ld->scalable_mode==SC_SPAT)
  {
    m->lower_layer_prediction_horizontal_size = _mpeg2Get_Bits(m,14);
    _mpeg2marker_bit(m,"sequence_scalable_extension()");
    m->lower_layer_prediction_vertical_size   = _mpeg2Get_Bits(m,14); 
    m->horizontal_subsampling_factor_m        = _mpeg2Get_Bits(m,5);
    m->horizontal_subsampling_factor_n        = _mpeg2Get_Bits(m,5);
    m->vertical_subsampling_factor_m          = _mpeg2Get_Bits(m,5);
    m->vertical_subsampling_factor_n          = _mpeg2Get_Bits(m,5);
  }

  if (m->ld->scalable_mode==SC_TEMP)
    _mpeg2Error(m, "temporal scalability not implemented\n");

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("sequence scalable extension (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SEQUENCE_LAYER)
    {
      printf("  scalable_mode=%d\n",ld->scalable_mode-1);
      printf("  layer_id=%d\n",layer_id);
      if (ld->scalable_mode==SC_SPAT)
      {
        printf("    lower_layer_prediction_horiontal_size=%d\n",
          lower_layer_prediction_horizontal_size);
        printf("    lower_layer_prediction_vertical_size=%d\n",
          lower_layer_prediction_vertical_size);
        printf("    horizontal_subsampling_factor_m=%d\n",
          horizontal_subsampling_factor_m);
        printf("    horizontal_subsampling_factor_n=%d\n",
          horizontal_subsampling_factor_n);
        printf("    vertical_subsampling_factor_m=%d\n",
          vertical_subsampling_factor_m);
        printf("    vertical_subsampling_factor_n=%d\n",
          vertical_subsampling_factor_n);
      }
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_sequence_scalable_extension++;
#endif /* VERIFY */

}


/* decode picture display extension */
/* ISO/IEC 13818-2 section 6.2.3.3. */
static void picture_display_extension(struct mpeg2obj *m)
{
  int i;
  int number_of_frame_center_offsets;
  int pos;

  pos = m->ld->Bitcnt;
  /* based on ISO/IEC 13818-2 section 6.3.12 
    (November 1994) Picture display extensions */

  /* derive number_of_frame_center_offsets */
  if(m->progressive_sequence)
  {
    if(m->repeat_first_field)
    {
      if(m->top_field_first)
        number_of_frame_center_offsets = 3;
      else
        number_of_frame_center_offsets = 2;
    }
    else
    {
      number_of_frame_center_offsets = 1;
    }
  }
  else
  {
    if(m->picture_structure!=FRAME_PICTURE)
    {
      number_of_frame_center_offsets = 1;
    }
    else
    {
      if(m->repeat_first_field)
        number_of_frame_center_offsets = 3;
      else
        number_of_frame_center_offsets = 2;
    }
  }


  /* now parse */
  for (i=0; i<number_of_frame_center_offsets; i++)
  {
    m->frame_center_horizontal_offset[i] = _mpeg2Get_Bits(m,16);
    _mpeg2marker_bit(m,"picture_display_extension, first marker bit");
    
    m->frame_center_vertical_offset[i]   = _mpeg2Get_Bits(m,16);
    _mpeg2marker_bit(m,"picture_display_extension, second marker bit");
  }

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("picture display extension (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SEQUENCE_LAYER)
    {

      for (i=0; i<number_of_frame_center_offsets; i++)
      {
        printf("  frame_center_horizontal_offset[%d]=%d\n",i,
          frame_center_horizontal_offset[i]);
        printf("  frame_center_vertical_offset[%d]=%d\n",i,
          frame_center_vertical_offset[i]);
      }
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_picture_display_extension++;
#endif /* VERIFY */

}


/* decode picture coding extension */
static void picture_coding_extension(struct mpeg2obj *m)
{
  int pos;

  pos = m->ld->Bitcnt;

  m->f_code[0][0] = _mpeg2Get_Bits(m,4);
  m->f_code[0][1] = _mpeg2Get_Bits(m,4);
  m->f_code[1][0] = _mpeg2Get_Bits(m,4);
  m->f_code[1][1] = _mpeg2Get_Bits(m,4);

  m->intra_dc_precision         = _mpeg2Get_Bits(m,2);
  m->picture_structure          = _mpeg2Get_Bits(m,2);
  m->top_field_first            = _mpeg2Get_Bits(m,1);
  m->frame_pred_frame_dct       = _mpeg2Get_Bits(m,1);
  m->concealment_motion_vectors = _mpeg2Get_Bits(m,1);
  m->ld->q_scale_type           = _mpeg2Get_Bits(m,1);
  m->intra_vlc_format           = _mpeg2Get_Bits(m,1);
  m->ld->alternate_scan         = _mpeg2Get_Bits(m,1);
  m->repeat_first_field         = _mpeg2Get_Bits(m,1);
  m->chroma_420_type            = _mpeg2Get_Bits(m,1);
  m->progressive_frame          = _mpeg2Get_Bits(m,1);
  m->composite_display_flag     = _mpeg2Get_Bits(m,1);

  if (m->composite_display_flag)
  {
    m->v_axis            = _mpeg2Get_Bits(m,1);
    m->field_sequence    = _mpeg2Get_Bits(m,3);
    m->sub_carrier       = _mpeg2Get_Bits(m,1);
    m->burst_amplitude   = _mpeg2Get_Bits(m,7);
    m->sub_carrier_phase = _mpeg2Get_Bits(m,8);
  }

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("picture coding extension (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SEQUENCE_LAYER)
    {
      printf("  forward horizontal f_code=%d\n", f_code[0][0]);
      printf("  forward vertical f_code=%d\n", f_code[0][1]);
      printf("  backward horizontal f_code=%d\n", f_code[1][0]);
      printf("  backward_vertical f_code=%d\n", f_code[1][1]);
      printf("  intra_dc_precision=%d\n",intra_dc_precision);
      printf("  picture_structure=%d\n",picture_structure);
      printf("  top_field_first=%d\n",top_field_first);
      printf("  frame_pred_frame_dct=%d\n",frame_pred_frame_dct);
      printf("  concealment_motion_vectors=%d\n",concealment_motion_vectors);
      printf("  q_scale_type=%d\n",ld->q_scale_type);
      printf("  intra_vlc_format=%d\n",intra_vlc_format);
      printf("  alternate_scan=%d\n",ld->alternate_scan);
      printf("  repeat_first_field=%d\n",repeat_first_field);
      printf("  chroma_420_type=%d\n",chroma_420_type);
      printf("  progressive_frame=%d\n",progressive_frame);
      printf("  composite_display_flag=%d\n",composite_display_flag);

      if (composite_display_flag)
      {
        printf("    v_axis=%d\n",v_axis);
        printf("    field_sequence=%d\n",field_sequence);
        printf("    sub_carrier=%d\n",sub_carrier);
        printf("    burst_amplitude=%d\n",burst_amplitude);
        printf("    sub_carrier_phase=%d\n",sub_carrier_phase);
      }
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_picture_coding_extension++;
#endif /* VERIFY */
}


/* decode picture spatial scalable extension */
/* ISO/IEC 13818-2 section 6.2.3.5. */
static void picture_spatial_scalable_extension(struct mpeg2obj *m)
{
  int pos;

  pos = m->ld->Bitcnt;

  m->ld->pict_scal = 1; /* use spatial scalability in this picture */

  m->lower_layer_temporal_reference = _mpeg2Get_Bits(m,10);
  _mpeg2marker_bit(m,"picture_spatial_scalable_extension(), first marker bit");
  m->lower_layer_horizontal_offset = _mpeg2Get_Bits(m,15);
  if (m->lower_layer_horizontal_offset>=16384)
    m->lower_layer_horizontal_offset-= 32768;
  _mpeg2marker_bit(m,"picture_spatial_scalable_extension(), second marker bit");
  m->lower_layer_vertical_offset = _mpeg2Get_Bits(m,15);
  if (m->lower_layer_vertical_offset>=16384)
    m->lower_layer_vertical_offset-= 32768;
  m->spatial_temporal_weight_code_table_index = _mpeg2Get_Bits(m,2);
  m->lower_layer_progressive_frame = _mpeg2Get_Bits(m,1);
  m->lower_layer_deinterlaced_field_select = _mpeg2Get_Bits(m,1);

#ifdef VERBOSE
  if (Verbose_Flag>NO_LAYER)
  {
    printf("picture spatial scalable extension (byte %d)\n",(pos>>3)-4);
    if (Verbose_Flag>SEQUENCE_LAYER)
    {
      printf("  lower_layer_temporal_reference=%d\n",lower_layer_temporal_reference);
      printf("  lower_layer_horizontal_offset=%d\n",lower_layer_horizontal_offset);
      printf("  lower_layer_vertical_offset=%d\n",lower_layer_vertical_offset);
      printf("  spatial_temporal_weight_code_table_index=%d\n",
        spatial_temporal_weight_code_table_index);
      printf("  lower_layer_progressive_frame=%d\n",lower_layer_progressive_frame);
      printf("  lower_layer_deinterlaced_field_select=%d\n",lower_layer_deinterlaced_field_select);
    }
  }
#endif /* VERBOSE */

#ifdef VERIFY
  verify_picture_spatial_scalable_extension++;
#endif /* VERIFY */

}


/* decode picture temporal scalable extension
 *
 * not implemented
 */
/* ISO/IEC 13818-2 section 6.2.3.4. */
static void picture_temporal_scalable_extension(struct mpeg2obj *m)
{
  _mpeg2Error(m, "temporal scalability not supported\n");

#ifdef VERIFY
  verify_picture_temporal_scalable_extension++;
#endif /* VERIFY */
}


/* decode extra bit information */
/* ISO/IEC 13818-2 section 6.2.3.4. */
static int extra_bit_information(struct mpeg2obj *m)
{
  int Byte_Count = 0;

  while (_mpeg2Get_Bits1(m))
  {
    _mpeg2Flush_Buffer(m,8);
    Byte_Count++;
  }

  return(Byte_Count);
}



/* ISO/IEC 13818-2 section 5.3 */
/* Purpose: this function is mainly designed to aid in bitstream conformance
   testing.  A simple Flush_Buffer(1) would do */
void _mpeg2marker_bit(struct mpeg2obj *m, const char *text)
{
  int marker;

  marker = _mpeg2Get_Bits(m,1);

#ifdef VERIFY  
  if(!marker)
    printf("ERROR: %s--marker_bit set to 0",text);
#endif
}


/* ISO/IEC 13818-2  sections 6.3.4.1 and 6.2.2.2.2 */
static void user_data(struct mpeg2obj *m)
{
  /* skip ahead to the next start code */
  _mpeg2next_start_code(m);
}



/* Copyright extension */
/* ISO/IEC 13818-2 section 6.2.3.6. */
/* (header added in November, 1994 to the IS document) */


static void copyright_extension(struct mpeg2obj *m)
{
  int pos;
  int reserved_data;

  pos = m->ld->Bitcnt;
  

  m->copyright_flag =       _mpeg2Get_Bits(m,1); 
  m->copyright_identifier = _mpeg2Get_Bits(m,8);
  m->original_or_copy =     _mpeg2Get_Bits(m,1);
  
  /* reserved */
  reserved_data = _mpeg2Get_Bits(m,7);

  _mpeg2marker_bit(m,"copyright_extension(), first marker bit");
  m->copyright_number_1 =   _mpeg2Get_Bits(m,20);
  _mpeg2marker_bit(m,"copyright_extension(), second marker bit");
  m->copyright_number_2 =   _mpeg2Get_Bits(m,22);
  _mpeg2marker_bit(m,"copyright_extension(), third marker bit");
  m->copyright_number_3 =   _mpeg2Get_Bits(m,22);

#ifdef VERIFY
  verify_copyright_extension++;
#endif /* VERIFY */
}



/* introduced in September 1995 to assist Spatial Scalability */
static void Update_Temporal_Reference_Tacking_Data(struct mpeg2obj *m)
{
  static int temporal_reference_wrap  = 0;
  static int temporal_reference_old   = 0;

  if (m->ld == &m->base)			/* *CH* */
  {
    if (m->picture_coding_type!=B_TYPE && m->temporal_reference!=temporal_reference_old) 	
    /* check first field of */
    {							
       /* non-B-frame */
      if (temporal_reference_wrap) 		
      {/* wrap occured at previous I- or P-frame */	
       /* now all intervening B-frames which could 
          still have high temporal_reference values are done  */
        m->Temporal_Reference_Base += 1024;
	    temporal_reference_wrap = 0;
      }
      
      /* distinguish from a reset */
      if (m->temporal_reference<temporal_reference_old && !m->Temporal_Reference_GOP_Reset)	
	    temporal_reference_wrap = 1;  /* we must have just passed a GOP-Header! */
      
      temporal_reference_old = m->temporal_reference;
      m->Temporal_Reference_GOP_Reset = 0;
    }

    m->True_Framenum = m->Temporal_Reference_Base + m->temporal_reference;
    
    /* temporary wrap of TR at 1024 for M frames */
    if (temporal_reference_wrap && m->temporal_reference <= temporal_reference_old)	
      m->True_Framenum += 1024;				

    m->True_Framenum_max = (m->True_Framenum > m->True_Framenum_max) ?
      m->True_Framenum : m->True_Framenum_max;
  }
}
