/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.tika;

import static org.junit.Assert.assertEquals;

import org.junit.Test;

public class TikaDetectionTest {

    private final Tika tika = new Tika();

    /**
     * This test checks that Tika correctly detects all the file extensions
     * defined in the mime.types file (revision 819245) of the Apache HTTP
     * Server project. The tests were created with:
     * <pre>
     * cat docs/conf/mime.types | grep -v '#' | perl -lne '/\S\s+\S/ and do {
     *     my ($type, @ext) = split /\s+/;
     *     for my $ext (@ext) {
     *         print "assertEquals(\"$type\", tika.detect(\"x.$ext\"));";
     *     }
     * }'
     * </pre>
     */
    @Test
    public void testHttpServerFileExtensions() {
        assertEquals("application/andrew-inset", tika.detect("x.ez"));
        assertEquals("application/applixware", tika.detect("x.aw"));
        assertEquals("application/atom+xml", tika.detect("x.atom"));
        assertEquals("application/atomcat+xml", tika.detect("x.atomcat"));
        assertEquals("application/atomsvc+xml", tika.detect("x.atomsvc"));
        assertEquals("application/ccxml+xml", tika.detect("x.ccxml"));
        assertEquals("application/cu-seeme", tika.detect("x.cu"));
        assertEquals("application/davmount+xml", tika.detect("x.davmount"));
        assertEquals("application/ecmascript", tika.detect("x.ecma"));
        assertEquals("application/emma+xml", tika.detect("x.emma"));
        assertEquals("application/epub+zip", tika.detect("x.epub"));
        assertEquals("application/font-tdpfr", tika.detect("x.pfr"));
        assertEquals("application/hyperstudio", tika.detect("x.stk"));
        assertEquals("application/java-archive", tika.detect("x.jar"));
        assertEquals("application/java-serialized-object", tika.detect("x.ser"));
        assertEquals("application/java-vm", tika.detect("x.class"));
        assertEquals("application/javascript", tika.detect("x.js"));
        assertEquals("application/json", tika.detect("x.json"));
        assertEquals("application/lost+xml", tika.detect("x.lostxml"));
        assertEquals("application/mac-binhex40", tika.detect("x.hqx"));
        assertEquals("application/mac-compactpro", tika.detect("x.cpt"));
        assertEquals("application/marc", tika.detect("x.mrc"));
        assertEquals("application/mathematica", tika.detect("x.ma"));
        assertEquals("application/mathematica", tika.detect("x.nb"));
        assertEquals("application/mathematica", tika.detect("x.mb"));
        assertEquals("application/mathml+xml", tika.detect("x.mathml"));
        assertEquals("application/mbox", tika.detect("x.mbox"));
        assertEquals("application/mediaservercontrol+xml", tika.detect("x.mscml"));
        assertEquals("application/mp4", tika.detect("x.mp4s"));
        assertEquals("application/msword", tika.detect("x.doc"));
        assertEquals("application/msword", tika.detect("x.dot"));
        assertEquals("application/mxf", tika.detect("x.mxf"));
        assertEquals("application/octet-stream", tika.detect("x.bin"));
        assertEquals("application/octet-stream", tika.detect("x.dms"));
        assertEquals("application/octet-stream", tika.detect("x.lha"));
        assertEquals("application/octet-stream", tika.detect("x.lrf"));
        assertEquals("application/octet-stream", tika.detect("x.lzh"));
        assertEquals("application/octet-stream", tika.detect("x.so"));
        assertEquals("application/octet-stream", tika.detect("x.iso"));
        assertEquals("application/octet-stream", tika.detect("x.dmg"));
        assertEquals("application/octet-stream", tika.detect("x.dist"));
        assertEquals("application/octet-stream", tika.detect("x.distz"));
        assertEquals("application/octet-stream", tika.detect("x.pkg"));
        assertEquals("application/octet-stream", tika.detect("x.bpk"));
        assertEquals("application/octet-stream", tika.detect("x.dump"));
        assertEquals("application/octet-stream", tika.detect("x.elc"));
        assertEquals("application/octet-stream", tika.detect("x.deploy"));
        assertEquals("application/oda", tika.detect("x.oda"));
        assertEquals("application/oebps-package+xml", tika.detect("x.opf"));
        assertEquals("application/ogg", tika.detect("x.ogx"));
        assertEquals("application/onenote", tika.detect("x.onetoc"));
        assertEquals("application/onenote", tika.detect("x.onetoc2"));
        assertEquals("application/onenote", tika.detect("x.onetmp"));
        assertEquals("application/onenote", tika.detect("x.onepkg"));
        assertEquals("application/patch-ops-error+xml", tika.detect("x.xer"));
        assertEquals("application/pdf", tika.detect("x.pdf"));
        assertEquals("application/pgp-encrypted", tika.detect("x.pgp"));
        assertEquals("application/pgp-signature", tika.detect("x.asc"));
        assertEquals("application/pgp-signature", tika.detect("x.sig"));
        assertEquals("application/pics-rules", tika.detect("x.prf"));
        assertEquals("application/pkcs10", tika.detect("x.p10"));
        assertEquals("application/pkcs7-mime", tika.detect("x.p7m"));
        assertEquals("application/pkcs7-mime", tika.detect("x.p7c"));
        assertEquals("application/pkcs7-signature", tika.detect("x.p7s"));
        assertEquals("application/pkix-cert", tika.detect("x.cer"));
        assertEquals("application/pkix-crl", tika.detect("x.crl"));
        assertEquals("application/pkix-pkipath", tika.detect("x.pkipath"));
        assertEquals("application/pkixcmp", tika.detect("x.pki"));
        assertEquals("application/pls+xml", tika.detect("x.pls"));
        assertEquals("application/illustrator", tika.detect("x.ai"));
        assertEquals("application/postscript", tika.detect("x.eps"));
        assertEquals("application/postscript", tika.detect("x.ps"));
        assertEquals("application/prs.cww", tika.detect("x.cww"));
        assertEquals("application/rdf+xml", tika.detect("x.rdf"));
        assertEquals("application/reginfo+xml", tika.detect("x.rif"));
        assertEquals("application/relax-ng-compact-syntax", tika.detect("x.rnc"));
        assertEquals("application/resource-lists+xml", tika.detect("x.rl"));
        assertEquals("application/resource-lists-diff+xml", tika.detect("x.rld"));
        assertEquals("application/rls-services+xml", tika.detect("x.rs"));
        assertEquals("application/rsd+xml", tika.detect("x.rsd"));
        assertEquals("application/rss+xml", tika.detect("x.rss"));
        assertEquals("application/rtf", tika.detect("x.rtf"));
        assertEquals("application/sbml+xml", tika.detect("x.sbml"));
        assertEquals("application/scvp-cv-request", tika.detect("x.scq"));
        assertEquals("application/scvp-cv-response", tika.detect("x.scs"));
        assertEquals("application/scvp-vp-request", tika.detect("x.spq"));
        assertEquals("application/scvp-vp-response", tika.detect("x.spp"));
        assertEquals("application/sdp", tika.detect("x.sdp"));
        assertEquals("application/set-payment-initiation", tika.detect("x.setpay"));
        assertEquals("application/set-registration-initiation", tika.detect("x.setreg"));
        assertEquals("application/sldworks", tika.detect("x.sldprt"));
        assertEquals("application/sldworks", tika.detect("x.sldasm"));
        assertEquals("application/sldworks", tika.detect("x.slddrw"));
        assertEquals("application/shf+xml", tika.detect("x.shf"));
        assertEquals("application/smil+xml", tika.detect("x.smi"));
        assertEquals("application/smil+xml", tika.detect("x.smil"));
        assertEquals("application/sparql-query", tika.detect("x.rq"));
        assertEquals("application/sparql-results+xml", tika.detect("x.srx"));
        assertEquals("application/srgs", tika.detect("x.gram"));
        assertEquals("application/srgs+xml", tika.detect("x.grxml"));
        assertEquals("application/ssml+xml", tika.detect("x.ssml"));
        assertEquals("application/vnd.3gpp.pic-bw-large", tika.detect("x.plb"));
        assertEquals("application/vnd.3gpp.pic-bw-small", tika.detect("x.psb"));
        assertEquals("application/vnd.3gpp.pic-bw-var", tika.detect("x.pvb"));
        assertEquals("application/vnd.3gpp2.tcap", tika.detect("x.tcap"));
        assertEquals("application/vnd.3m.post-it-notes", tika.detect("x.pwn"));
        assertEquals("application/vnd.accpac.simply.aso", tika.detect("x.aso"));
        assertEquals("application/vnd.accpac.simply.imp", tika.detect("x.imp"));
        assertEquals("application/vnd.acucobol", tika.detect("x.acu"));
        assertEquals("application/vnd.acucorp", tika.detect("x.atc"));
        assertEquals("application/vnd.acucorp", tika.detect("x.acutc"));
        assertEquals("application/vnd.adobe.air-application-installer-package+zip", tika.detect("x.air"));
        assertEquals("application/vnd.adobe.xdp+xml", tika.detect("x.xdp"));
        assertEquals("application/vnd.adobe.xfdf", tika.detect("x.xfdf"));
        assertEquals("application/vnd.airzip.filesecure.azf", tika.detect("x.azf"));
        assertEquals("application/vnd.airzip.filesecure.azs", tika.detect("x.azs"));
        assertEquals("application/vnd.amazon.ebook", tika.detect("x.azw"));
        assertEquals("application/vnd.americandynamics.acc", tika.detect("x.acc"));
        assertEquals("application/vnd.amiga.ami", tika.detect("x.ami"));
        assertEquals("application/vnd.android.package-archive", tika.detect("x.apk"));
        assertEquals("application/vnd.anser-web-certificate-issue-initiation", tika.detect("x.cii"));
        assertEquals("application/vnd.anser-web-funds-transfer-initiation", tika.detect("x.fti"));
        assertEquals("application/vnd.antix.game-component", tika.detect("x.atx"));
        assertEquals("application/vnd.apple.installer+xml", tika.detect("x.mpkg"));
        assertEquals("application/vnd.arastra.swi", tika.detect("x.swi"));
        // Differ from httpd - Adobe After Effects is a much more common user of .AEP these days
        //assertEquals("application/vnd.audiograph", tika.detect("x.aep"));
        assertEquals("application/vnd.blueice.multipass", tika.detect("x.mpm"));
        assertEquals("application/vnd.bmi", tika.detect("x.bmi"));
        assertEquals("application/vnd.businessobjects", tika.detect("x.rep"));
        assertEquals("application/vnd.chemdraw+xml", tika.detect("x.cdxml"));
        assertEquals("application/vnd.chipnuts.karaoke-mmd", tika.detect("x.mmd"));
        assertEquals("application/vnd.cinderella", tika.detect("x.cdy"));
        assertEquals("application/vnd.claymore", tika.detect("x.cla"));
        assertEquals("application/vnd.clonk.c4group", tika.detect("x.c4g"));
        assertEquals("application/vnd.clonk.c4group", tika.detect("x.c4d"));
        assertEquals("application/vnd.clonk.c4group", tika.detect("x.c4f"));
        assertEquals("application/vnd.clonk.c4group", tika.detect("x.c4p"));
        assertEquals("application/vnd.clonk.c4group", tika.detect("x.c4u"));
        assertEquals("application/vnd.commonspace", tika.detect("x.csp"));
        assertEquals("application/vnd.contact.cmsg", tika.detect("x.cdbcmsg"));
        assertEquals("application/vnd.cosmocaller", tika.detect("x.cmc"));
        assertEquals("application/vnd.crick.clicker", tika.detect("x.clkx"));
        assertEquals("application/vnd.crick.clicker.keyboard", tika.detect("x.clkk"));
        assertEquals("application/vnd.crick.clicker.palette", tika.detect("x.clkp"));
        assertEquals("application/vnd.crick.clicker.template", tika.detect("x.clkt"));
        assertEquals("application/vnd.crick.clicker.wordbank", tika.detect("x.clkw"));
        assertEquals("application/vnd.criticaltools.wbs+xml", tika.detect("x.wbs"));
        assertEquals("application/vnd.ctc-posml", tika.detect("x.pml"));
        assertEquals("application/vnd.cups-ppd", tika.detect("x.ppd"));
        assertEquals("application/vnd.curl.car", tika.detect("x.car"));
        assertEquals("application/vnd.curl.pcurl", tika.detect("x.pcurl"));
        assertEquals("application/vnd.data-vision.rdz", tika.detect("x.rdz"));
        assertEquals("application/vnd.denovo.fcselayout-link", tika.detect("x.fe_launch"));
        assertEquals("application/vnd.dna", tika.detect("x.dna"));
        assertEquals("application/vnd.dolby.mlp", tika.detect("x.mlp"));
        assertEquals("application/vnd.dpgraph", tika.detect("x.dpg"));
        assertEquals("application/vnd.dreamfactory", tika.detect("x.dfac"));
        assertEquals("application/vnd.dynageo", tika.detect("x.geo"));
        assertEquals("application/vnd.ecowin.chart", tika.detect("x.mag"));
        assertEquals("application/vnd.enliven", tika.detect("x.nml"));
        assertEquals("application/vnd.epson.esf", tika.detect("x.esf"));
        assertEquals("application/vnd.epson.msf", tika.detect("x.msf"));
        assertEquals("application/vnd.epson.quickanime", tika.detect("x.qam"));
        assertEquals("application/vnd.epson.salt", tika.detect("x.slt"));
        assertEquals("application/vnd.epson.ssf", tika.detect("x.ssf"));
        assertEquals("application/vnd.eszigno3+xml", tika.detect("x.es3"));
        assertEquals("application/vnd.eszigno3+xml", tika.detect("x.et3"));
        assertEquals("application/vnd.ezpix-album", tika.detect("x.ez2"));
        assertEquals("application/vnd.ezpix-package", tika.detect("x.ez3"));
        assertEquals("application/vnd.fdf", tika.detect("x.fdf"));
        assertEquals("application/vnd.fdsn.mseed", tika.detect("x.mseed"));
        assertEquals("application/vnd.fdsn.seed", tika.detect("x.seed"));
        assertEquals("application/vnd.fdsn.seed", tika.detect("x.dataless"));
        assertEquals("application/vnd.flographit", tika.detect("x.gph"));
        assertEquals("application/vnd.fluxtime.clip", tika.detect("x.ftc"));
        assertEquals("application/vnd.framemaker", tika.detect("x.fm"));
        assertEquals("application/vnd.framemaker", tika.detect("x.frame"));
        assertEquals("application/vnd.framemaker", tika.detect("x.maker"));
        assertEquals("application/vnd.framemaker", tika.detect("x.book"));
        assertEquals("application/vnd.frogans.fnc", tika.detect("x.fnc"));
        assertEquals("application/vnd.frogans.ltf", tika.detect("x.ltf"));
        assertEquals("application/vnd.fsc.weblaunch", tika.detect("x.fsc"));
        assertEquals("application/vnd.fujitsu.oasys", tika.detect("x.oas"));
        assertEquals("application/vnd.fujitsu.oasys2", tika.detect("x.oa2"));
        assertEquals("application/vnd.fujitsu.oasys3", tika.detect("x.oa3"));
        assertEquals("application/vnd.fujitsu.oasysgp", tika.detect("x.fg5"));
        assertEquals("application/vnd.fujitsu.oasysprs", tika.detect("x.bh2"));
        assertEquals("application/vnd.fujixerox.ddd", tika.detect("x.ddd"));
        assertEquals("application/vnd.fujixerox.docuworks", tika.detect("x.xdw"));
        assertEquals("application/vnd.fujixerox.docuworks.binder", tika.detect("x.xbd"));
        assertEquals("application/vnd.fuzzysheet", tika.detect("x.fzs"));
        assertEquals("application/vnd.genomatix.tuxedo", tika.detect("x.txd"));
        assertEquals("application/vnd.geogebra.file", tika.detect("x.ggb"));
        assertEquals("application/vnd.geogebra.tool", tika.detect("x.ggt"));
        assertEquals("application/vnd.geometry-explorer", tika.detect("x.gex"));
        assertEquals("application/vnd.geometry-explorer", tika.detect("x.gre"));
        assertEquals("application/vnd.gmx", tika.detect("x.gmx"));
        assertEquals("application/vnd.google-earth.kml+xml", tika.detect("x.kml"));
        assertEquals("application/vnd.google-earth.kmz", tika.detect("x.kmz"));
        assertEquals("application/vnd.grafeq", tika.detect("x.gqf"));
        assertEquals("application/vnd.grafeq", tika.detect("x.gqs"));
        assertEquals("application/vnd.groove-account", tika.detect("x.gac"));
        assertEquals("application/vnd.groove-help", tika.detect("x.ghf"));
        assertEquals("application/vnd.groove-identity-message", tika.detect("x.gim"));
        assertEquals("application/vnd.groove-injector", tika.detect("x.grv"));
        assertEquals("application/vnd.groove-tool-message", tika.detect("x.gtm"));
        assertEquals("application/vnd.groove-tool-template", tika.detect("x.tpl"));
        assertEquals("application/vnd.groove-vcard", tika.detect("x.vcg"));
        assertEquals("application/vnd.handheld-entertainment+xml", tika.detect("x.zmm"));
        assertEquals("application/vnd.hbci", tika.detect("x.hbci"));
        assertEquals("application/vnd.hhe.lesson-player", tika.detect("x.les"));
        assertEquals("application/vnd.hp-hpgl", tika.detect("x.hpgl"));
        assertEquals("application/vnd.hp-hpid", tika.detect("x.hpid"));
        assertEquals("application/vnd.hp-hps", tika.detect("x.hps"));
        assertEquals("application/vnd.hp-jlyt", tika.detect("x.jlt"));
        assertEquals("application/vnd.hp-pcl", tika.detect("x.pcl"));
        assertEquals("application/vnd.hp-pclxl", tika.detect("x.pclxl"));
        assertEquals("application/vnd.hydrostatix.sof-data", tika.detect("x.sfd-hdstx"));
        assertEquals("application/vnd.hzn-3d-crossword", tika.detect("x.x3d"));
        assertEquals("application/vnd.ibm.minipay", tika.detect("x.mpy"));
        assertEquals("application/vnd.ibm.modcap", tika.detect("x.afp"));
        assertEquals("application/vnd.ibm.modcap", tika.detect("x.listafp"));
        assertEquals("application/vnd.ibm.modcap", tika.detect("x.list3820"));
        assertEquals("application/vnd.ibm.rights-management", tika.detect("x.irm"));
        assertEquals("application/vnd.ibm.secure-container", tika.detect("x.sc"));
        assertEquals("application/vnd.iccprofile", tika.detect("x.icc"));
        assertEquals("application/vnd.iccprofile", tika.detect("x.icm"));
        assertEquals("application/vnd.igloader", tika.detect("x.igl"));
        assertEquals("application/vnd.immervision-ivp", tika.detect("x.ivp"));
        assertEquals("application/vnd.immervision-ivu", tika.detect("x.ivu"));
        assertEquals("application/vnd.intercon.formnet", tika.detect("x.xpw"));
        assertEquals("application/vnd.intercon.formnet", tika.detect("x.xpx"));
        assertEquals("application/vnd.intu.qbo", tika.detect("x.qbo"));
        assertEquals("application/vnd.intu.qfx", tika.detect("x.qfx"));
        assertEquals("application/vnd.ipunplugged.rcprofile", tika.detect("x.rcprofile"));
        assertEquals("application/vnd.irepository.package+xml", tika.detect("x.irp"));
        assertEquals("application/vnd.is-xpr", tika.detect("x.xpr"));
        assertEquals("application/vnd.jam", tika.detect("x.jam"));
        assertEquals("application/vnd.jcp.javame.midlet-rms", tika.detect("x.rms"));
        assertEquals("application/vnd.jisp", tika.detect("x.jisp"));
        assertEquals("application/vnd.joost.joda-archive", tika.detect("x.joda"));
        assertEquals("application/vnd.kahootz", tika.detect("x.ktz"));
        assertEquals("application/vnd.kahootz", tika.detect("x.ktr"));
        assertEquals("application/vnd.kde.karbon", tika.detect("x.karbon"));
        assertEquals("application/vnd.kde.kchart", tika.detect("x.chrt"));
        assertEquals("application/vnd.kde.kformula", tika.detect("x.kfo"));
        assertEquals("application/vnd.kde.kivio", tika.detect("x.flw"));
        assertEquals("application/vnd.kde.kontour", tika.detect("x.kon"));
        assertEquals("application/vnd.kde.kpresenter", tika.detect("x.kpr"));
        assertEquals("application/vnd.kde.kpresenter", tika.detect("x.kpt"));
        assertEquals("application/vnd.kde.kspread", tika.detect("x.ksp"));
        assertEquals("application/vnd.kde.kword", tika.detect("x.kwd"));
        assertEquals("application/vnd.kde.kword", tika.detect("x.kwt"));
        assertEquals("application/vnd.kenameaapp", tika.detect("x.htke"));
        assertEquals("application/vnd.kidspiration", tika.detect("x.kia"));
        assertEquals("application/vnd.kinar", tika.detect("x.kne"));
        assertEquals("application/vnd.kinar", tika.detect("x.knp"));
        assertEquals("application/vnd.koan", tika.detect("x.skp"));
        assertEquals("application/vnd.koan", tika.detect("x.skd"));
        assertEquals("application/vnd.koan", tika.detect("x.skt"));
        assertEquals("application/vnd.koan", tika.detect("x.skm"));
        assertEquals("application/vnd.kodak-descriptor", tika.detect("x.sse"));
        assertEquals("application/vnd.llamagraphics.life-balance.desktop", tika.detect("x.lbd"));
        assertEquals("application/vnd.llamagraphics.life-balance.exchange+xml", tika.detect("x.lbe"));
        assertEquals("application/vnd.lotus-1-2-3", tika.detect("x.123"));
        assertEquals("application/vnd.lotus-approach", tika.detect("x.apr"));
        assertEquals("application/vnd.lotus-freelance", tika.detect("x.pre"));
        assertEquals("application/vnd.lotus-notes", tika.detect("x.nsf"));
        assertEquals("application/vnd.lotus-organizer", tika.detect("x.org"));
        assertEquals("text/x-scheme", tika.detect("x.scm"));
        assertEquals("application/vnd.lotus-wordpro", tika.detect("x.lwp"));
        assertEquals("application/vnd.macports.portpkg", tika.detect("x.portpkg"));
        assertEquals("application/vnd.mcd", tika.detect("x.mcd"));
        assertEquals("application/vnd.medcalcdata", tika.detect("x.mc1"));
        assertEquals("application/vnd.mediastation.cdkey", tika.detect("x.cdkey"));
        assertEquals("application/vnd.mfer", tika.detect("x.mwf"));
        assertEquals("application/vnd.mfmp", tika.detect("x.mfm"));
        assertEquals("application/vnd.micrografx.flo", tika.detect("x.flo"));
        assertEquals("application/vnd.micrografx.igx", tika.detect("x.igx"));
        assertEquals("application/vnd.mif", tika.detect("x.mif"));
        assertEquals("application/vnd.mobius.daf", tika.detect("x.daf"));
        assertEquals("application/vnd.mobius.dis", tika.detect("x.dis"));
        assertEquals("application/vnd.mobius.mbk", tika.detect("x.mbk"));
        assertEquals("application/vnd.mobius.mqy", tika.detect("x.mqy"));
        assertEquals("application/vnd.mobius.msl", tika.detect("x.msl"));
        assertEquals("application/vnd.mobius.plc", tika.detect("x.plc"));
        assertEquals("application/vnd.mobius.txf", tika.detect("x.txf"));
        assertEquals("application/vnd.mophun.application", tika.detect("x.mpn"));
        assertEquals("application/vnd.mophun.certificate", tika.detect("x.mpc"));
        assertEquals("application/vnd.mozilla.xul+xml", tika.detect("x.xul"));
        assertEquals("application/vnd.ms-artgalry", tika.detect("x.cil"));
        assertEquals("application/vnd.ms-cab-compressed", tika.detect("x.cab"));
        assertEquals("application/vnd.ms-excel", tika.detect("x.xls"));
        assertEquals("application/vnd.ms-excel", tika.detect("x.xlm"));
        assertEquals("application/vnd.ms-excel", tika.detect("x.xla"));
        assertEquals("application/vnd.ms-excel", tika.detect("x.xlc"));
        assertEquals("application/vnd.ms-excel", tika.detect("x.xlt"));
        assertEquals("application/vnd.ms-excel", tika.detect("x.xlw"));
        assertEquals("application/vnd.ms-excel.addin.macroenabled.12", tika.detect("x.xlam"));
        assertEquals("application/vnd.ms-excel.sheet.binary.macroenabled.12", tika.detect("x.xlsb"));
        assertEquals("application/vnd.ms-excel.sheet.macroenabled.12", tika.detect("x.xlsm"));
        assertEquals("application/vnd.ms-excel.template.macroenabled.12", tika.detect("x.xltm"));
        assertEquals("application/vnd.ms-fontobject", tika.detect("x.eot"));
        assertEquals("application/vnd.ms-htmlhelp", tika.detect("x.chm"));
        assertEquals("application/vnd.ms-ims", tika.detect("x.ims"));
        assertEquals("application/vnd.ms-lrm", tika.detect("x.lrm"));
        assertEquals("application/vnd.ms-pki.seccat", tika.detect("x.cat"));
        assertEquals("application/vnd.ms-pki.stl", tika.detect("x.stl"));
        assertEquals("application/vnd.ms-powerpoint", tika.detect("x.ppt"));
        assertEquals("application/vnd.ms-powerpoint", tika.detect("x.pps"));
        assertEquals("application/vnd.ms-powerpoint", tika.detect("x.pot"));
        assertEquals("application/vnd.ms-powerpoint.addin.macroenabled.12", tika.detect("x.ppam"));
        assertEquals("application/vnd.ms-powerpoint.presentation.macroenabled.12", tika.detect("x.pptm"));
        assertEquals("application/vnd.ms-powerpoint.slide.macroenabled.12", tika.detect("x.sldm"));
        assertEquals("application/vnd.ms-powerpoint.slideshow.macroenabled.12", tika.detect("x.ppsm"));
        assertEquals("application/vnd.ms-powerpoint.template.macroenabled.12", tika.detect("x.potm"));
        assertEquals("application/vnd.ms-project", tika.detect("x.mpp"));
        assertEquals("application/vnd.ms-project", tika.detect("x.mpt"));
        assertEquals("application/vnd.ms-word.document.macroenabled.12", tika.detect("x.docm"));
        assertEquals("application/vnd.ms-word.template.macroenabled.12", tika.detect("x.dotm"));
        assertEquals("application/vnd.ms-works", tika.detect("x.wps"));
        assertEquals("application/vnd.ms-works", tika.detect("x.wks"));
        assertEquals("application/vnd.ms-works", tika.detect("x.wcm"));
        assertEquals("application/vnd.ms-works", tika.detect("x.wdb"));
        assertEquals("application/vnd.ms-wpl", tika.detect("x.wpl"));
        assertEquals("application/vnd.ms-xpsdocument", tika.detect("x.xps"));
        assertEquals("application/vnd.mseq", tika.detect("x.mseq"));
        assertEquals("application/vnd.musician", tika.detect("x.mus"));
        assertEquals("application/vnd.muvee.style", tika.detect("x.msty"));
        assertEquals("application/vnd.neurolanguage.nlu", tika.detect("x.nlu"));
        assertEquals("application/vnd.noblenet-directory", tika.detect("x.nnd"));
        assertEquals("application/vnd.noblenet-sealer", tika.detect("x.nns"));
        assertEquals("application/vnd.noblenet-web", tika.detect("x.nnw"));
        assertEquals("application/vnd.nokia.n-gage.data", tika.detect("x.ngdat"));
        assertEquals("application/vnd.nokia.n-gage.symbian.install", tika.detect("x.n-gage"));
        assertEquals("application/vnd.nokia.radio-preset", tika.detect("x.rpst"));
        assertEquals("application/vnd.nokia.radio-presets", tika.detect("x.rpss"));
        assertEquals("application/vnd.novadigm.edm", tika.detect("x.edm"));
        assertEquals("application/vnd.novadigm.edx", tika.detect("x.edx"));
        assertEquals("application/vnd.novadigm.ext", tika.detect("x.ext"));
        assertEquals("application/vnd.oasis.opendocument.chart", tika.detect("x.odc"));
        assertEquals("application/vnd.oasis.opendocument.chart-template", tika.detect("x.otc"));
        assertEquals("application/vnd.oasis.opendocument.database", tika.detect("x.odb"));
        assertEquals("application/vnd.oasis.opendocument.formula", tika.detect("x.odf"));
        assertEquals("application/vnd.oasis.opendocument.formula-template", tika.detect("x.odft"));
        assertEquals("application/vnd.oasis.opendocument.graphics", tika.detect("x.odg"));
        assertEquals("application/vnd.oasis.opendocument.graphics-template", tika.detect("x.otg"));
        assertEquals("application/vnd.oasis.opendocument.image", tika.detect("x.odi"));
        assertEquals("application/vnd.oasis.opendocument.image-template", tika.detect("x.oti"));
        assertEquals("application/vnd.oasis.opendocument.presentation", tika.detect("x.odp"));
        assertEquals("application/vnd.oasis.opendocument.presentation-template", tika.detect("x.otp"));
        assertEquals("application/vnd.oasis.opendocument.spreadsheet", tika.detect("x.ods"));
        assertEquals("application/vnd.oasis.opendocument.spreadsheet-template", tika.detect("x.ots"));
        assertEquals("application/vnd.oasis.opendocument.text", tika.detect("x.odt"));
        assertEquals("application/vnd.oasis.opendocument.text-master", tika.detect("x.otm"));
        assertEquals("application/vnd.oasis.opendocument.text-template", tika.detect("x.ott"));
        assertEquals("application/vnd.oasis.opendocument.text-web", tika.detect("x.oth"));
        assertEquals("application/vnd.olpc-sugar", tika.detect("x.xo"));
        assertEquals("application/vnd.oma.dd2+xml", tika.detect("x.dd2"));
        assertEquals("application/vnd.openofficeorg.extension", tika.detect("x.oxt"));
        assertEquals("application/vnd.openxmlformats-officedocument.presentationml.presentation", tika.detect("x.pptx"));
        assertEquals("application/vnd.openxmlformats-officedocument.presentationml.slide", tika.detect("x.sldx"));
        assertEquals("application/vnd.openxmlformats-officedocument.presentationml.slideshow", tika.detect("x.ppsx"));
        assertEquals("application/vnd.openxmlformats-officedocument.presentationml.template", tika.detect("x.potx"));
        assertEquals("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", tika.detect("x.xlsx"));
        assertEquals("application/vnd.openxmlformats-officedocument.spreadsheetml.template", tika.detect("x.xltx"));
        assertEquals("application/vnd.openxmlformats-officedocument.wordprocessingml.document", tika.detect("x.docx"));
        assertEquals("application/vnd.openxmlformats-officedocument.wordprocessingml.template", tika.detect("x.dotx"));
        assertEquals("application/vnd.osgi.dp", tika.detect("x.dp"));
        assertEquals("chemical/x-pdb", tika.detect("x.pdb"));
        assertEquals("application/vnd.palm", tika.detect("x.pqa"));
        assertEquals("application/vnd.palm", tika.detect("x.oprc"));
        assertEquals("application/vnd.pg.format", tika.detect("x.str"));
        assertEquals("application/vnd.pg.osasli", tika.detect("x.ei6"));
        assertEquals("application/vnd.picsel", tika.detect("x.efif"));
        assertEquals("application/vnd.pocketlearn", tika.detect("x.plf"));
        assertEquals("application/vnd.powerbuilder6", tika.detect("x.pbd"));
        assertEquals("application/vnd.previewsystems.box", tika.detect("x.box"));
        assertEquals("application/vnd.proteus.magazine", tika.detect("x.mgz"));
        assertEquals("application/vnd.publishare-delta-tree", tika.detect("x.qps"));
        assertEquals("application/vnd.pvi.ptid1", tika.detect("x.ptid"));
        assertEquals("application/vnd.quark.quarkxpress", tika.detect("x.qxd"));
        assertEquals("application/vnd.quark.quarkxpress", tika.detect("x.qxt"));
        assertEquals("application/vnd.quark.quarkxpress", tika.detect("x.qwd"));
        assertEquals("application/vnd.quark.quarkxpress", tika.detect("x.qwt"));
        assertEquals("application/vnd.quark.quarkxpress", tika.detect("x.qxl"));
        assertEquals("application/vnd.quark.quarkxpress", tika.detect("x.qxb"));
        assertEquals("application/vnd.recordare.musicxml", tika.detect("x.mxl"));
        assertEquals("application/vnd.recordare.musicxml+xml", tika.detect("x.musicxml"));
        assertEquals("application/vnd.rim.cod", tika.detect("x.cod"));
        assertEquals("application/vnd.rn-realmedia", tika.detect("x.rm"));
        assertEquals("application/vnd.route66.link66+xml", tika.detect("x.link66"));
        assertEquals("application/vnd.seemail", tika.detect("x.see"));
        assertEquals("application/vnd.sema", tika.detect("x.sema"));
        assertEquals("application/vnd.semd", tika.detect("x.semd"));
        assertEquals("application/vnd.semf", tika.detect("x.semf"));
        assertEquals("application/vnd.shana.informed.formdata", tika.detect("x.ifm"));
        assertEquals("application/vnd.shana.informed.formtemplate", tika.detect("x.itp"));
        assertEquals("application/vnd.shana.informed.interchange", tika.detect("x.iif"));
        assertEquals("application/vnd.shana.informed.package", tika.detect("x.ipk"));
        assertEquals("application/vnd.simtech-mindmapper", tika.detect("x.twd"));
        assertEquals("application/vnd.simtech-mindmapper", tika.detect("x.twds"));
        assertEquals("application/vnd.smaf", tika.detect("x.mmf"));
        assertEquals("application/vnd.smart.teacher", tika.detect("x.teacher"));
        assertEquals("application/vnd.solent.sdkm+xml", tika.detect("x.sdkm"));
        assertEquals("application/vnd.solent.sdkm+xml", tika.detect("x.sdkd"));
        assertEquals("application/vnd.spotfire.dxp", tika.detect("x.dxp"));
        assertEquals("application/vnd.spotfire.sfs", tika.detect("x.sfs"));
        assertEquals("application/vnd.stardivision.calc", tika.detect("x.sdc"));
        assertEquals("application/vnd.stardivision.draw", tika.detect("x.sda"));
        assertEquals("application/vnd.stardivision.impress", tika.detect("x.sdd"));
        assertEquals("application/vnd.stardivision.math", tika.detect("x.smf"));
        assertEquals("application/vnd.stardivision.writer", tika.detect("x.sdw"));
        assertEquals("application/x-staroffice-template", tika.detect("x.vor"));
        assertEquals("application/vnd.stardivision.writer-global", tika.detect("x.sgl"));
        assertEquals("application/vnd.sun.xml.calc", tika.detect("x.sxc"));
        assertEquals("application/vnd.sun.xml.calc.template", tika.detect("x.stc"));
        assertEquals("application/vnd.sun.xml.draw", tika.detect("x.sxd"));
        assertEquals("application/vnd.sun.xml.draw.template", tika.detect("x.std"));
        assertEquals("application/vnd.sun.xml.impress", tika.detect("x.sxi"));
        assertEquals("application/vnd.sun.xml.impress.template", tika.detect("x.sti"));
        assertEquals("application/vnd.sun.xml.math", tika.detect("x.sxm"));
        assertEquals("application/vnd.sun.xml.writer", tika.detect("x.sxw"));
        assertEquals("application/vnd.sun.xml.writer.global", tika.detect("x.sxg"));
        assertEquals("application/vnd.sun.xml.writer.template", tika.detect("x.stw"));
        assertEquals("application/vnd.sus-calendar", tika.detect("x.sus"));
        assertEquals("application/vnd.sus-calendar", tika.detect("x.susp"));
        assertEquals("application/vnd.svd", tika.detect("x.svd"));
        assertEquals("application/vnd.symbian.install", tika.detect("x.sis"));
        assertEquals("application/vnd.symbian.install", tika.detect("x.sisx"));
        assertEquals("application/vnd.syncml+xml", tika.detect("x.xsm"));
        assertEquals("application/vnd.syncml.dm+wbxml", tika.detect("x.bdm"));
        assertEquals("application/vnd.syncml.dm+xml", tika.detect("x.xdm"));
        assertEquals("application/vnd.tao.intent-module-archive", tika.detect("x.tao"));
        assertEquals("application/vnd.tmobile-livetv", tika.detect("x.tmo"));
        assertEquals("application/vnd.trid.tpt", tika.detect("x.tpt"));
        assertEquals("application/vnd.triscape.mxs", tika.detect("x.mxs"));
        assertEquals("application/vnd.trueapp", tika.detect("x.tra"));
        assertEquals("application/vnd.ufdl", tika.detect("x.ufd"));
        assertEquals("application/vnd.ufdl", tika.detect("x.ufdl"));
        assertEquals("application/vnd.uiq.theme", tika.detect("x.utz"));
        assertEquals("application/vnd.umajin", tika.detect("x.umj"));
        assertEquals("application/vnd.unity", tika.detect("x.unityweb"));
        assertEquals("application/vnd.uoml+xml", tika.detect("x.uoml"));
        assertEquals("application/vnd.vcx", tika.detect("x.vcx"));
        assertEquals("application/vnd.visio", tika.detect("x.vsd"));
        assertEquals("application/vnd.visio", tika.detect("x.vst"));
        assertEquals("application/vnd.visio", tika.detect("x.vss"));
        assertEquals("application/vnd.visio", tika.detect("x.vsw"));
        assertEquals("application/vnd.visionary", tika.detect("x.vis"));
        assertEquals("application/vnd.vsf", tika.detect("x.vsf"));
        assertEquals("application/vnd.wap.wbxml", tika.detect("x.wbxml"));
        assertEquals("application/vnd.wap.wmlc", tika.detect("x.wmlc"));
        assertEquals("application/vnd.wap.wmlscriptc", tika.detect("x.wmlsc"));
        assertEquals("application/vnd.webturbo", tika.detect("x.wtb"));
        assertEquals("application/vnd.wordperfect", tika.detect("x.wpd"));
        assertEquals("application/vnd.wqd", tika.detect("x.wqd"));
        assertEquals("application/vnd.wt.stf", tika.detect("x.stf"));
        assertEquals("application/vnd.xara", tika.detect("x.xar"));
        assertEquals("application/vnd.xfdl", tika.detect("x.xfdl"));
        assertEquals("application/vnd.yamaha.hv-dic", tika.detect("x.hvd"));
        assertEquals("application/vnd.yamaha.hv-script", tika.detect("x.hvs"));
        assertEquals("application/vnd.yamaha.hv-voice", tika.detect("x.hvp"));
        assertEquals("application/vnd.yamaha.openscoreformat", tika.detect("x.osf"));
        assertEquals("application/vnd.yamaha.openscoreformat.osfpvg+xml", tika.detect("x.osfpvg"));
        assertEquals("application/vnd.yamaha.smaf-audio", tika.detect("x.saf"));
        assertEquals("application/vnd.yamaha.smaf-phrase", tika.detect("x.spf"));
        assertEquals("application/vnd.yellowriver-custom-menu", tika.detect("x.cmp"));
        assertEquals("application/vnd.zul", tika.detect("x.zir"));
        assertEquals("application/vnd.zul", tika.detect("x.zirz"));
        assertEquals("application/vnd.zzazz.deck+xml", tika.detect("x.zaz"));
        assertEquals("application/voicexml+xml", tika.detect("x.vxml"));
        assertEquals("application/winhlp", tika.detect("x.hlp"));
        assertEquals("application/wsdl+xml", tika.detect("x.wsdl"));
        assertEquals("application/wspolicy+xml", tika.detect("x.wspolicy"));
        assertEquals("application/x-abiword", tika.detect("x.abw"));
        assertEquals("application/x-ace-compressed", tika.detect("x.ace"));
        assertEquals("application/x-authorware-bin", tika.detect("x.aab"));
        assertEquals("application/x-authorware-bin", tika.detect("x.x32"));
        assertEquals("application/x-authorware-bin", tika.detect("x.u32"));
        assertEquals("application/x-authorware-bin", tika.detect("x.vox"));
        assertEquals("application/x-authorware-map", tika.detect("x.aam"));
        assertEquals("application/x-authorware-seg", tika.detect("x.aas"));
        assertEquals("application/x-bcpio", tika.detect("x.bcpio"));
        assertEquals("application/x-bittorrent", tika.detect("x.torrent"));
        assertEquals("application/x-bzip", tika.detect("x.bz"));
        assertEquals("application/x-bzip2", tika.detect("x.bz2"));
        assertEquals("application/x-bzip2", tika.detect("x.boz"));
        assertEquals("application/x-cdlink", tika.detect("x.vcd"));
        assertEquals("application/x-chat", tika.detect("x.chat"));
        assertEquals("application/x-chess-pgn", tika.detect("x.pgn"));
        assertEquals("application/x-cpio", tika.detect("x.cpio"));
        assertEquals("application/x-csh", tika.detect("x.csh"));
        assertEquals("application/x-debian-package", tika.detect("x.deb"));
        assertEquals("application/x-debian-package", tika.detect("x.udeb"));
        assertEquals("application/x-director", tika.detect("x.dir"));
        assertEquals("application/x-director", tika.detect("x.dcr"));
        assertEquals("application/x-director", tika.detect("x.dxr"));
        assertEquals("application/x-director", tika.detect("x.cst"));
        assertEquals("application/x-director", tika.detect("x.cct"));
        assertEquals("application/x-director", tika.detect("x.cxt"));
        assertEquals("application/x-director", tika.detect("x.w3d"));
        assertEquals("application/x-director", tika.detect("x.fgd"));
        assertEquals("application/x-director", tika.detect("x.swa"));
        assertEquals("application/x-doom", tika.detect("x.wad"));
        assertEquals("application/x-dtbncx+xml", tika.detect("x.ncx"));
        assertEquals("application/x-dtbook+xml", tika.detect("x.dtb"));
        assertEquals("application/x-dtbresource+xml", tika.detect("x.res"));
        assertEquals("application/x-dvi", tika.detect("x.dvi"));
        assertEquals("application/x-font-bdf", tika.detect("x.bdf"));
        assertEquals("application/x-font-ghostscript", tika.detect("x.gsf"));
        assertEquals("application/x-font-linux-psf", tika.detect("x.psf"));
        assertEquals("application/x-font-otf", tika.detect("x.otf"));
        assertEquals("application/x-font-pcf", tika.detect("x.pcf"));
        assertEquals("application/x-font-snf", tika.detect("x.snf"));
        assertEquals("application/x-font-ttf", tika.detect("x.ttf"));
        assertEquals("application/x-font-ttf", tika.detect("x.ttc"));
        assertEquals("application/x-font-type1", tika.detect("x.pfa"));
        assertEquals("application/x-font-type1", tika.detect("x.pfb"));
        // TODO Get these fixed upstream too
        //assertEquals("application/x-font-type1", tika.detect("x.pfm"));
        //assertEquals("application/x-font-type1", tika.detect("x.afm"));
        assertEquals("application/x-font-printer-metric", tika.detect("x.pfm"));
        assertEquals("application/x-font-adobe-metric", tika.detect("x.afm"));
        assertEquals("application/x-futuresplash", tika.detect("x.spl"));
        assertEquals("application/x-gnumeric", tika.detect("x.gnumeric"));
        assertEquals("application/x-gtar", tika.detect("x.gtar"));
        assertEquals("application/x-hdf", tika.detect("x.hdf"));
        assertEquals("application/x-java-jnlp-file", tika.detect("x.jnlp"));
        assertEquals("application/x-latex", tika.detect("x.latex"));
        assertEquals("application/x-mobipocket-ebook", tika.detect("x.prc"));
        assertEquals("application/x-mobipocket-ebook", tika.detect("x.mobi"));
        assertEquals("application/x-ms-application", tika.detect("x.application"));
        assertEquals("application/x-ms-wmd", tika.detect("x.wmd"));
        assertEquals("application/x-ms-wmz", tika.detect("x.wmz"));
        assertEquals("application/x-ms-xbap", tika.detect("x.xbap"));
        assertEquals("application/x-msaccess", tika.detect("x.mdb"));
        assertEquals("application/x-msbinder", tika.detect("x.obd"));
        assertEquals("application/x-mscardfile", tika.detect("x.crd"));
        assertEquals("application/x-msclip", tika.detect("x.clp"));
        assertEquals("application/x-dosexec", tika.detect("x.exe"));
        assertEquals("application/x-msdownload", tika.detect("x.dll"));
        assertEquals("application/x-msdownload", tika.detect("x.com"));
        assertEquals("application/x-msdownload", tika.detect("x.bat"));
        assertEquals("application/x-msdownload", tika.detect("x.msi"));
        assertEquals("application/x-msmediaview", tika.detect("x.mvb"));
        assertEquals("application/x-msmediaview", tika.detect("x.m13"));
        assertEquals("application/x-msmediaview", tika.detect("x.m14"));
        assertEquals("application/x-msmetafile", tika.detect("x.wmf"));
        assertEquals("application/x-msmoney", tika.detect("x.mny"));
        assertEquals("application/x-mspublisher", tika.detect("x.pub"));
        assertEquals("application/x-msschedule", tika.detect("x.scd"));
        assertEquals("application/x-msterminal", tika.detect("x.trm"));
        assertEquals("application/x-mswrite", tika.detect("x.wri"));
        assertEquals("application/x-netcdf", tika.detect("x.nc"));
        assertEquals("application/x-netcdf", tika.detect("x.cdf"));
        assertEquals("application/x-pkcs12", tika.detect("x.p12"));
        assertEquals("application/x-pkcs12", tika.detect("x.pfx"));
        assertEquals("application/x-pkcs7-certificates", tika.detect("x.p7b"));
        assertEquals("application/x-pkcs7-certificates", tika.detect("x.spc"));
        assertEquals("application/x-pkcs7-certreqresp", tika.detect("x.p7r"));
        assertEquals("application/x-rar-compressed", tika.detect("x.rar"));
        assertEquals("application/x-sh", tika.detect("x.sh"));
        assertEquals("application/x-shar", tika.detect("x.shar"));
        assertEquals("application/x-shockwave-flash", tika.detect("x.swf"));
        assertEquals("application/x-silverlight-app", tika.detect("x.xap"));
        assertEquals("application/x-stuffit", tika.detect("x.sit"));
        assertEquals("application/x-stuffitx", tika.detect("x.sitx"));
        assertEquals("application/x-sv4cpio", tika.detect("x.sv4cpio"));
        assertEquals("application/x-sv4crc", tika.detect("x.sv4crc"));
        assertEquals("application/x-tar", tika.detect("x.tar"));
        assertEquals("text/x-tcl", tika.detect("x.tcl"));
        assertEquals("application/x-tex", tika.detect("x.tex"));
        assertEquals("application/x-tex-tfm", tika.detect("x.tfm"));
        assertEquals("application/x-texinfo", tika.detect("x.texinfo"));
        assertEquals("application/x-texinfo", tika.detect("x.texi"));
        assertEquals("application/x-ustar", tika.detect("x.ustar"));
        assertEquals("application/x-wais-source", tika.detect("x.src"));
        assertEquals("application/x-x509-ca-cert", tika.detect("x.der"));
        assertEquals("application/x-x509-ca-cert", tika.detect("x.crt"));
        assertEquals("application/x-xfig", tika.detect("x.fig"));
        assertEquals("application/x-xpinstall", tika.detect("x.xpi"));
        assertEquals("application/xenc+xml", tika.detect("x.xenc"));
        assertEquals("application/xhtml+xml", tika.detect("x.xhtml"));
        assertEquals("application/xhtml+xml", tika.detect("x.xht"));
        assertEquals("application/xml", tika.detect("x.xml"));
        assertEquals("application/xml", tika.detect("x.xsl"));
        assertEquals("application/xml-dtd", tika.detect("x.dtd"));
        assertEquals("application/xop+xml", tika.detect("x.xop"));
        assertEquals("application/xslt+xml", tika.detect("x.xslt"));
        assertEquals("application/xspf+xml", tika.detect("x.xspf"));
        assertEquals("application/xv+xml", tika.detect("x.mxml"));
        assertEquals("application/xv+xml", tika.detect("x.xhvml"));
        assertEquals("application/xv+xml", tika.detect("x.xvml"));
        assertEquals("application/xv+xml", tika.detect("x.xvm"));
        assertEquals("application/zip", tika.detect("x.zip"));
        assertEquals("audio/adpcm", tika.detect("x.adp"));
        assertEquals("audio/basic", tika.detect("x.au"));
        assertEquals("audio/basic", tika.detect("x.snd"));
        assertEquals("audio/midi", tika.detect("x.mid"));
        assertEquals("audio/midi", tika.detect("x.midi"));
        assertEquals("audio/midi", tika.detect("x.kar"));
        assertEquals("audio/midi", tika.detect("x.rmi"));
        assertEquals("audio/mp4", tika.detect("x.mp4a"));
        assertEquals("audio/mpeg", tika.detect("x.mpga"));
        assertEquals("audio/mpeg", tika.detect("x.mp2"));
        assertEquals("audio/mpeg", tika.detect("x.mp2a"));
        assertEquals("audio/mpeg", tika.detect("x.mp3"));
        assertEquals("audio/mpeg", tika.detect("x.m2a"));
        assertEquals("audio/mpeg", tika.detect("x.m3a"));
        assertEquals("audio/ogg", tika.detect("x.oga"));
        assertEquals("audio/ogg", tika.detect("x.ogg"));
        assertEquals("audio/ogg", tika.detect("x.spx"));
        assertEquals("audio/vnd.digital-winds", tika.detect("x.eol"));
        assertEquals("audio/vnd.dts", tika.detect("x.dts"));
        assertEquals("audio/vnd.dts.hd", tika.detect("x.dtshd"));
        assertEquals("audio/vnd.lucent.voice", tika.detect("x.lvp"));
        assertEquals("audio/vnd.ms-playready.media.pya", tika.detect("x.pya"));
        assertEquals("audio/vnd.nuera.ecelp4800", tika.detect("x.ecelp4800"));
        assertEquals("audio/vnd.nuera.ecelp7470", tika.detect("x.ecelp7470"));
        assertEquals("audio/vnd.nuera.ecelp9600", tika.detect("x.ecelp9600"));
        assertEquals("audio/x-aac", tika.detect("x.aac"));
        assertEquals("audio/x-aiff", tika.detect("x.aif"));
        assertEquals("audio/x-aiff", tika.detect("x.aiff"));
        assertEquals("audio/x-aiff", tika.detect("x.aifc"));
        assertEquals("audio/x-mpegurl", tika.detect("x.m3u"));
        assertEquals("audio/x-ms-wax", tika.detect("x.wax"));
        assertEquals("audio/x-ms-wma", tika.detect("x.wma"));
        assertEquals("audio/x-pn-realaudio", tika.detect("x.ram"));
        assertEquals("audio/x-pn-realaudio", tika.detect("x.ra"));
        assertEquals("audio/x-pn-realaudio-plugin", tika.detect("x.rmp"));
        assertEquals("audio/x-wav", tika.detect("x.wav"));
        assertEquals("chemical/x-cdx", tika.detect("x.cdx"));
        assertEquals("chemical/x-cif", tika.detect("x.cif"));
        assertEquals("chemical/x-cmdf", tika.detect("x.cmdf"));
        assertEquals("chemical/x-cml", tika.detect("x.cml"));
        assertEquals("chemical/x-csml", tika.detect("x.csml"));
        assertEquals("chemical/x-xyz", tika.detect("x.xyz"));
        assertEquals("image/x-ms-bmp", tika.detect("x.bmp"));
        assertEquals("image/cgm", tika.detect("x.cgm"));
        assertEquals("image/g3fax", tika.detect("x.g3"));
        assertEquals("image/gif", tika.detect("x.gif"));
        assertEquals("image/ief", tika.detect("x.ief"));
        assertEquals("image/jpeg", tika.detect("x.jpeg"));
        assertEquals("image/jpeg", tika.detect("x.jpg"));
        assertEquals("image/jpeg", tika.detect("x.jpe"));
        assertEquals("image/jpm", tika.detect("x.jpm"));
        assertEquals("image/jpm", tika.detect("x.jpgm"));
        assertEquals("image/png", tika.detect("x.png"));
        assertEquals("image/prs.btif", tika.detect("x.btif"));
        assertEquals("image/svg+xml", tika.detect("x.svg"));
        assertEquals("image/svg+xml", tika.detect("x.svgz"));
        assertEquals("image/tiff", tika.detect("x.tiff"));
        assertEquals("image/tiff", tika.detect("x.tif"));
        assertEquals("image/vnd.adobe.photoshop", tika.detect("x.psd"));
        assertEquals("image/vnd.djvu", tika.detect("x.djvu"));
        assertEquals("image/vnd.djvu", tika.detect("x.djv"));
        assertEquals("image/vnd.dwg", tika.detect("x.dwg"));
        assertEquals("image/vnd.dxf", tika.detect("x.dxf"));
        assertEquals("image/vnd.fastbidsheet", tika.detect("x.fbs"));
        assertEquals("image/vnd.fpx", tika.detect("x.fpx"));
        assertEquals("image/vnd.fst", tika.detect("x.fst"));
        assertEquals("image/vnd.fujixerox.edmics-mmr", tika.detect("x.mmr"));
        assertEquals("image/vnd.fujixerox.edmics-rlc", tika.detect("x.rlc"));
        assertEquals("image/vnd.ms-modi", tika.detect("x.mdi"));
        assertEquals("image/vnd.net-fpx", tika.detect("x.npx"));
        assertEquals("image/vnd.wap.wbmp", tika.detect("x.wbmp"));
        assertEquals("image/vnd.xiff", tika.detect("x.xif"));
        assertEquals("image/x-cmu-raster", tika.detect("x.ras"));
        assertEquals("image/x-cmx", tika.detect("x.cmx"));
        assertEquals("image/x-freehand", tika.detect("x.fh"));
        assertEquals("image/x-freehand", tika.detect("x.fhc"));
        assertEquals("image/x-freehand", tika.detect("x.fh4"));
        assertEquals("image/x-freehand", tika.detect("x.fh5"));
        assertEquals("image/x-freehand", tika.detect("x.fh7"));
        // Differ from httpd - An official mimetype has subsequently been issued
        //  favicon.ico +friends should now be image/vnd.microsoft.icon
        //assertEquals("image/x-icon", tika.detect("x.ico"));
        assertEquals("image/x-pcx", tika.detect("x.pcx"));
        assertEquals("image/x-pict", tika.detect("x.pic"));
        assertEquals("image/x-pict", tika.detect("x.pct"));
        assertEquals("image/x-portable-anymap", tika.detect("x.pnm"));
        assertEquals("image/x-portable-bitmap", tika.detect("x.pbm"));
        assertEquals("image/x-portable-graymap", tika.detect("x.pgm"));
        assertEquals("image/x-portable-pixmap", tika.detect("x.ppm"));
        assertEquals("image/x-rgb", tika.detect("x.rgb"));
        assertEquals("image/x-xbitmap", tika.detect("x.xbm"));
        assertEquals("image/x-xpixmap", tika.detect("x.xpm"));
        assertEquals("image/x-xwindowdump", tika.detect("x.xwd"));
        assertEquals("message/rfc822", tika.detect("x.eml"));
        assertEquals("message/rfc822", tika.detect("x.mime"));
        assertEquals("model/iges", tika.detect("x.igs"));
        assertEquals("model/iges", tika.detect("x.iges"));
        assertEquals("model/mesh", tika.detect("x.msh"));
        assertEquals("model/mesh", tika.detect("x.mesh"));
        assertEquals("model/mesh", tika.detect("x.silo"));
        assertEquals("model/vnd.dwf", tika.detect("x.dwf"));
        assertEquals("model/vnd.gdl", tika.detect("x.gdl"));
        assertEquals("model/vnd.gtw", tika.detect("x.gtw"));
        assertEquals("model/vnd.mts", tika.detect("x.mts"));
        assertEquals("model/vnd.vtu", tika.detect("x.vtu"));
        assertEquals("model/vrml", tika.detect("x.wrl"));
        assertEquals("model/vrml", tika.detect("x.vrml"));
        assertEquals("text/calendar", tika.detect("x.ics"));
        assertEquals("text/calendar", tika.detect("x.ifb"));
        assertEquals("text/css", tika.detect("x.css"));
        assertEquals("text/csv", tika.detect("x.csv"));
        assertEquals("text/html", tika.detect("x.html"));
        assertEquals("text/html", tika.detect("x.htm"));
        assertEquals("text/plain", tika.detect("x.txt"));
        assertEquals("text/plain", tika.detect("x.text"));
        assertEquals("text/plain", tika.detect("x.conf"));
        assertEquals("text/plain", tika.detect("x.def"));
        assertEquals("text/plain", tika.detect("x.list"));
        assertEquals("text/x-log", tika.detect("x.log"));
        assertEquals("text/plain", tika.detect("x.in"));
        assertEquals("text/prs.lines.tag", tika.detect("x.dsc"));
        assertEquals("text/richtext", tika.detect("x.rtx"));
        assertEquals("text/sgml", tika.detect("x.sgml"));
        assertEquals("text/sgml", tika.detect("x.sgm"));
        assertEquals("text/tab-separated-values", tika.detect("x.tsv"));
        assertEquals("text/troff", tika.detect("x.t"));
        assertEquals("text/troff", tika.detect("x.tr"));
        assertEquals("text/troff", tika.detect("x.roff"));
        assertEquals("text/troff", tika.detect("x.man"));
        assertEquals("text/troff", tika.detect("x.me"));
        assertEquals("text/troff", tika.detect("x.ms"));
        assertEquals("text/uri-list", tika.detect("x.uri"));
        assertEquals("text/uri-list", tika.detect("x.uris"));
        assertEquals("text/uri-list", tika.detect("x.urls"));
        assertEquals("text/vnd.curl", tika.detect("x.curl"));
        assertEquals("text/vnd.curl.dcurl", tika.detect("x.dcurl"));
        assertEquals("text/vnd.curl.scurl", tika.detect("x.scurl"));
        assertEquals("text/vnd.curl.mcurl", tika.detect("x.mcurl"));
        assertEquals("text/vnd.fly", tika.detect("x.fly"));
        assertEquals("text/vnd.fmi.flexstor", tika.detect("x.flx"));
        assertEquals("text/vnd.graphviz", tika.detect("x.gv"));
        assertEquals("text/vnd.in3d.3dml", tika.detect("x.3dml"));
        assertEquals("text/vnd.in3d.spot", tika.detect("x.spot"));
        assertEquals("text/vnd.sun.j2me.app-descriptor", tika.detect("x.jad"));
        assertEquals("text/vnd.wap.wml", tika.detect("x.wml"));
        assertEquals("text/vnd.wap.wmlscript", tika.detect("x.wmls"));
        assertEquals("text/x-assembly", tika.detect("x.s"));
        assertEquals("text/x-assembly", tika.detect("x.asm"));
        assertEquals("text/x-csrc", tika.detect("x.c"));
        assertEquals("text/x-c++src", tika.detect("x.cc"));
        assertEquals("text/x-c++src", tika.detect("x.cxx"));
        assertEquals("text/x-c++src", tika.detect("x.cpp"));
        assertEquals("text/x-chdr", tika.detect("x.h"));
        assertEquals("text/x-c++hdr", tika.detect("x.hh"));
        assertEquals("text/x-fortran", tika.detect("x.f"));
        assertEquals("text/x-fortran", tika.detect("x.for"));
        assertEquals("text/x-fortran", tika.detect("x.f77"));
        assertEquals("text/x-fortran", tika.detect("x.f90"));
        assertEquals("text/x-pascal", tika.detect("x.p"));
        assertEquals("text/x-pascal", tika.detect("x.pas"));
        assertEquals("text/x-java-source", tika.detect("x.java"));
        assertEquals("text/x-setext", tika.detect("x.etx"));
        assertEquals("text/x-uuencode", tika.detect("x.uu"));
        assertEquals("text/x-vcalendar", tika.detect("x.vcs"));
        assertEquals("text/x-vcard", tika.detect("x.vcf"));
        assertEquals("video/3gpp", tika.detect("x.3gp"));
        assertEquals("video/3gpp2", tika.detect("x.3g2"));
        assertEquals("video/h261", tika.detect("x.h261"));
        assertEquals("video/h263", tika.detect("x.h263"));
        assertEquals("video/h264", tika.detect("x.h264"));
        assertEquals("video/jpeg", tika.detect("x.jpgv"));
        assertEquals("video/mj2", tika.detect("x.mj2"));
        assertEquals("video/mj2", tika.detect("x.mjp2"));
        assertEquals("video/mp4", tika.detect("x.mp4"));
        assertEquals("video/mp4", tika.detect("x.mp4v"));
        assertEquals("video/mp4", tika.detect("x.mpg4"));
        assertEquals("video/mpeg", tika.detect("x.mpeg"));
        assertEquals("video/mpeg", tika.detect("x.mpg"));
        assertEquals("video/mpeg", tika.detect("x.mpe"));
        assertEquals("video/mpeg", tika.detect("x.m1v"));
        assertEquals("video/mpeg", tika.detect("x.m2v"));
        assertEquals("video/ogg", tika.detect("x.ogv"));
        assertEquals("video/quicktime", tika.detect("x.qt"));
        assertEquals("video/quicktime", tika.detect("x.mov"));
        assertEquals("video/vnd.fvt", tika.detect("x.fvt"));
        assertEquals("video/vnd.mpegurl", tika.detect("x.mxu"));
        assertEquals("video/vnd.mpegurl", tika.detect("x.m4u"));
        assertEquals("video/vnd.ms-playready.media.pyv", tika.detect("x.pyv"));
        assertEquals("video/vnd.vivo", tika.detect("x.viv"));
        assertEquals("video/x-f4v", tika.detect("x.f4v"));
        assertEquals("video/x-fli", tika.detect("x.fli"));
        assertEquals("video/x-flv", tika.detect("x.flv"));
        assertEquals("video/x-m4v", tika.detect("x.m4v"));
        assertEquals("video/x-ms-asf", tika.detect("x.asf"));
        assertEquals("video/x-ms-asf", tika.detect("x.asx"));
        assertEquals("video/x-ms-wm", tika.detect("x.wm"));
        assertEquals("video/x-ms-wmv", tika.detect("x.wmv"));
        assertEquals("video/x-ms-wmx", tika.detect("x.wmx"));
        assertEquals("video/x-ms-wvx", tika.detect("x.wvx"));
        assertEquals("video/x-msvideo", tika.detect("x.avi"));
        assertEquals("video/x-sgi-movie", tika.detect("x.movie"));
        assertEquals("x-conference/x-cooltalk", tika.detect("x.ice"));
    }

}
