use ui;
use graphics;
use core:geometry;

/**
 * A workspace, where we place all variables and other data structures currently
 * visible to the user.
 */
class Workspace on Render {
	init() {
		init() {
			enableThreadStep = true;
			trackDeadlock = true;
		}

		hints << DefaultViewHints();
	}

	// Hints on how to draw things.
	ViewHints[] hints;

	// Should we track memory accesses?
	private Bool trackMemAccess;

	// Should we look for deadlocks? Reset when we have found one.
	public Bool trackDeadlock;

	// Track memory?
	public assign trackMemory(Bool v) {
		trackMemAccess = v;
		if (!v) {
			for (e in elements) {
				if (data = e.visual.contained as Data)
					data.clearAccess();
			}

			for (obj, data in stackObjMap) {
				data.clearAccess();
			}
		}
	}

	// Allow stepping in threads.
	private Bool enableThreadStep;

	// Allow stepping.
	public assign enableStep(Bool v) {
		enableThreadStep = v;
		for (e in elements) {
			if (t = e.visual.contained as ThreadData) {
				t.enableStep = v;
			}
		}
		v;
	}


	// Draw the workspace. Assumes we can clip anything not inside 'viewport'.
	void render(Rect viewport, Graphics g) {
		// Find new docked elements.
		for (e in elements) {
			if (!e.docked) {
				if (e.visual.docked) {
					e.docked = true;
					docked.push(e);
				}
			}
		}

		layoutDocked(viewport);

		for (e in elements) {
			// TODO: Clipping?
			if (!e.topmost) {
				var visual = e.visual;
				visual.draw(g, active is visual);
			}
		}

		// Draw elements that are topmost.
		for (i, e in elements) {
			if (e.topmost) {
				var visual = e.visual;
				visual.draw(g, active is visual);
			}
		}

		// Draw pointers etc above all other elements.
		for (i, e in elements) {
			var visual = e.visual;
			visual.drawLinks(g, active is visual);
		}
	}

	// Layout docked elements.
	private void layoutDocked(Rect viewport) {
		Float space = 10;
		Float x = space + viewport.p0.x;

		for (Nat i = 0; i < docked.count; i++) {
			var e = docked[i];
			// Note: e.docked is set to false when the element is removed.
			if (!e.visual.docked | !e.docked) {
				e.docked = false;
				docked.remove(i);
				i--;
				continue;
			}

			if (e.thread) {
				Size sz = e.visual.size;
				e.visual.pos.x = x;
				e.visual.pos.y = viewport.p1.y - space - sz.h;

				x += sz.w + space;
			}
			if (e.output) {
				Size sz = e.visual.size;
				e.visual.pos.x = viewport.p1.x - sz.w - space;
				// e.visual.pos.y = viewport.p1.y - sz.h - space;
				e.visual.pos.y = viewport.p0.y + space;
			}
		}
	}

	// Helper for 'findAt', only considers either topmost or non-topmost elements.
	private Drawable? findAt(Point pt, Bool activate, Bool topmost) {
		for (Nat i = elements.count; i > 0; i--) {
			var elem = elements[i-1];
			if (elem.topmost != topmost)
				continue;

			Drawable d = elem.visual;
			if (d.rect.contains(pt)) {
				if (activate) {
					active = d;

					// Move it last to make it get focus.
					elements.remove(i-1);
					elements.push(elem);
				}
				return d.findAt(pt);
			}
		}

		return null;
	}

	// Find an object at the specified location.
	private Drawable? findAt(Point pt, Bool activate) {
		if (r = findAt(pt, activate, true))
			return r;
		if (r = findAt(pt, activate, false))
			return r;

		active = null;
		return null;
	}
	Drawable? findAt(Point pt) { findAt(pt, false); }
	Drawable? findAtActivate(Point pt) { findAt(pt, true); }
	void deactivate() { active = null; }

	/**
	 * Output.
	 */

	// Our output object, if we created it.
	private OutputText? output;

	// Get the output, create it if it does not exist already.
	private OutputText getOutput() {
		if (output)
			return output;

		OutputText created;
		Element elem(Dockable(created, "Output"));
		elem.output = true;
		elem.topmost = true;
		elements << elem;
		output = created;
		created;
	}

	void outputText(Str text) {
		var out = getOutput();
		out.add(text);
	}


	/**
	 * Globals.
	 */

	// The globals object, if we created it.
	private Globals? globals;

	// Set new global variables.
	assign globals(core:lang:GlobalVar[] vars) {
		globals = null;

		if (vars.empty)
			return;

		Globals created(vars);
		DataWithCaption caption(created, "Globals");
		Element elem(unsafe:RawPtr(created), caption);
		elements << elem;

		caption.pos = Point(20, 20);
		globals = created;
	}


	/**
	 * Object traversal.
	 */

	// Notify the workspace that we will start traversing objects now.
	void beginTraversal() {
		objMap.clear();
		stackObjMap.clear();
		work.clear();
		maxPos = Point();
		for (i, d in elements) {
			d.visited = false;
			if (d.object.any) {
				objMap.put(d.object, i);
			}
			if (!d.thread & !d.output) {
				maxPos.x = max(maxPos.x, d.visual.rect.p1.x);
				// maxPos.y = max(maxPos.y, d.visual.rect.p1.y);
			}
		}

		// Mark globals as visited, if applicable.
		if (globals) {
			traverseNoCreate(unsafe:RawPtr(globals));
		}
	}

	// Recursively add an object to the workspace.
	Data traverse(unsafe:RawPtr object) {
		traverse(object, null);
	}

	// Provide hints when traversing.
	Data traverse(unsafe:RawPtr object, CreateHints? hints) {
		// See if it exists.
		if (data = traverseNoCreate(object))
			return data;

		// print("Miss: ${object} was not found.");

		// Create a new instance.
		Nat id = elements.count;
		var visual = createData(this, object);
		if (hints)
			visual = hints.decorate(visual);
		elements << Element(object, visual);
		objMap.put(object, id);

		layout(visual);

		// Follow pointers.
		work.push(WorkItem(visual, object));

		return visual;
	}

	// Traverse, but never attempt to create new nodes from 'object', as it might refer to stack
	// objects or similar. This means that we will only traverse 'object' if we manage to verify
	// that it is indeed a valid pointer.
	Data? traverseNoCreate(unsafe:RawPtr object) {
		// Handle stack allocations.
		if (stackObjMap.has(object))
			return stackObjMap.get(object);

		// Heap allocations.
		Nat id = objMap.get(object, elements.count);
		if (id < elements.count) {
			// Update the old instance if it is not yet traversed.
			var elem = elements[id];
			if (data = elem.visual as Data) {
				// Note: If we get here, we know that 'object' is an acceptable reference to this
				// instance, and as such it is safe to use it to update the element if we need to do
				// so.
				if (!elem.visited) {
					elem.visited = true;
					work.push(WorkItem(data, object));
				}
				return data;
			}
		}

		null;
	}

	// Traverse the current contents of a thread.
	void traverse(ProgThread thread) {
		unsafe:RawPtr key(thread);
		Nat id = objMap.get(key, elements.count);
		WorkList list(work, stackObjMap);
		if (id < elements.count) {
			// Update the old one if possible.
			var elem = elements[id];
			elem.visited = true;

			Drawable data = elem.visual.contained;
			if (data as ThreadData) {
				data.traverse(this, list);
			}
		} else {
			ThreadData t(thread);
			t.enableStep = enableThreadStep;
			t.traverse(this, list);

			elements << Element(thread, Dockable(t, "Thread ${thread.threadId}").allowResize.anchorBottom);
			objMap.put(key, id);
		}
	}

	// Traverse an element. Check if it is a thread or a regular object first.
	void traverse(Element elem) {
		if (elem.object.empty) {
			elem.visited = true;
		} else if (elem.thread) {
			if (t = elem.object.asTObject() as ProgThread)
				traverse(t);
		} else {
			traverse(elem.object);
		}
	}

	// Notify the workspace that traversal is over, and that any remaning objects may be removed.
	ProgramError? endTraversal() {
		popQueue();

		// Find any dead objects now, and potentially traverse them as well.
		for (elem in elements) {
			if (!elem.visited) {
				// Ask it if it is ready to be removed.
				if (!elem.visual.remove)
					// Traverse to keep dependencies alive.
					traverse(elem);
			}
		}

		popQueue();

		// Clean up any remaining objects.
		Nat to = 0;
		for (Nat i = 0; i < elements.count; i++) {
			Element here = elements[i];
			if (to < i)
				elements[to] = here;
			if (here.visited)
				to++;
			else
				here.docked = false;
		}
		while (to < elements.count) {
			elements.pop();
		}

		ProgramError? error;

		// Mark reads/writes.
		if (trackMemAccess)
			error = updateMemAccess();

		// Detect deadlocks.
		if (error.empty & trackDeadlock)
			error = findDeadlock();

		return error;
	}

	// Remove everything in the workspace.
	void clear() {
		elements.clear();
		docked.clear();
		output = null;
		globals = null;
	}

private:
	// A single element on the screen, with some additional metadata.
	class Element on Render {
		// Which object this corresponds to, if any.
		unsafe:RawPtr object;

		// Visual part of the object.
		Drawable visual;

		// Was this visited during a traversal?
		Bool visited;

		// Is this a thread? (impacts layout etc.)
		Bool thread;

		// Is this an output panel? (Put on the top right).
		Bool output;

		// Always on top?
		Bool topmost;

		// Is the element docked? (i.e. is it in the "docked"-array?)
		Bool docked;

		// Create, attach to an object.
		init(unsafe:RawPtr object, Drawable visual) {
			init {
				object = object;
				visual = visual;
				visited = true;
				topmost = false;
			}
		}

		// Create, for a thread.
		init(ProgThread thread, Drawable visual) {
			init {
				object = unsafe:RawPtr(thread);
				visual = visual;
				visited = true;
				thread = true;
				topmost = true;
			}
		}

		// Create independet object.
		init(Drawable visual) {
			init {
				visual = visual;
			}
		}
	}

	// All elements in the workspace, in drawing order (back to front).
	Element[] elements;

	// Docked elements, to keep track of the order they were docked in.
	Element[] docked;

	// Active drawable.
	Drawable? active;

	// Map of object pointers to the corresponding data elements. Only valid during an object traversal.
	unsafe:RawPtr->Nat objMap;

	// Map of object pointers to objects on the stack of a thread.
	unsafe:RawPtr->Data stackObjMap;

	// Queue for BFS traversals when we traverse the object graph.
	Queue<WorkItem> work;

	// Basic layout during traversal: Try to put things into a "tree", where the direct children of
	// an object are placed to the right of the object, and otherwise below other objects.
	Point maxPos;

	// Traverse the work queue.
	void popQueue() {
		// Pop the queue until it is empty.
		while (work.any) {
			WorkItem top = work.top;
			top.data.update(this, top.object, 0);
			work.pop();
		}
	}

	// Perform layout of a newly created object based on the state of the traversal.
	private void layout(Data visual) {
		// Look at the current topmost element in the queue (if any), and update maxX and maxY
		// accordingly.
		if (work.any) {
			// Don't examine objects that are not on the heap.
			if (objMap.has(work.top.object)) {
				var top = work.top.data.rect;

				// New column?
				if (top.p1.x > maxPos.x) {
					maxPos.x = top.p1.x;
					maxPos.y = 0;
				}
			}
		}

		// Put it at an appropriate location.
		Point pos = maxPos + layoutSpace;
		visual.pos = pos;
		maxPos.y = pos.y + visual.size.h;
	}

	// Update memory accesses in all objects.
	private ProgramError? updateMemAccess() {
		// First, collect all MemTracker objects.
		Trackers trackers;
		for (e in elements) {
			if (t = e.visual.contained as ThreadData) {
				trackers.put(t);
			}
		}

		for (e in elements) {
			if (data = e.visual.contained as Data)
				data.updateMemAccess(e.object, trackers);
		}

		for (obj, data in stackObjMap) {
			data.updateMemAccess(obj, trackers);
		}

		if (trackers.errors.any) {
			pauseThreads();

			return DataRaceError(trackers.errors);
		}

		return null;
	}

	// See if all threads seem to wait for some kind of synchronization primitive, and report a
	// deadlock in that case. We don't have perfect knowledge of what threads are waiting for, so
	// this is the best we can do.
	private ProgramError? findDeadlock() {
		Bool anyThreads = false;
		Bool allSleeping = true;

		for (e in elements) {
			if (t = e.visual.contained as ThreadData) {
				if (t.thread.alive) {
					anyThreads = true;
					allSleeping &= t.thread.sleeping;
				}
			}
		}

		if (anyThreads & allSleeping) {
			// Inhibit further messages until reset.
			trackDeadlock = false;
			return DeadlockError();
		}

		return null;
	}

	// Pause all threads due to some error.
	private void pauseThreads() {
		for (e in elements) {
			if (t = e.visual.contained as ThreadData) {
				t.thread.pause();
			}
		}
	}

	// Create a new node for an object. Called by 'traverse'.
	public Data createData(unsafe:RawPtr object) {
		if (t = object.type) {
			TypeInfo info(t, true, false);
			Data d = createData(info, object, 0);
			if (d as DataWithCaption)
				return d;
			addTitle(d, info);
		} else {
			DataWithCaption(UnknownData(), "?");
		}
	}

	// Create data for an element on the stack. Respects if it was a pointer or not.
	public Data createStackData(unsafe:RawPtr object) {
		if (t = object.type) {
			TypeInfo info(t, object.isValue, false);
			if (!info.isValue) {
				StackPointer();
			} else {
				createData(info, object, 0);
			}
		} else {
			// It is probably a pointer that's NULL at the moment.
			StackPointer();
		}
	}

	// Create a new node for some object that is a part of another object.
	public Data createData(TypeInfo info, unsafe:RawPtr object, Nat offset) {
		for (h in hints) {
			if (d = h.createData(this, info, object, offset)) {
				if (d as Composite) {
					return addTitle(d, info);
				} else {
					return d;
				}
			}
		}

		return UnknownData();
	}

	Data addTitle(Data d, TypeInfo info) {
		DataWithCaption(d, typeTitle(info));
	}

	// Compute a title for a Type.
	public Str typeTitle(TypeInfo type) {
		for (h in hints) {
			if (r = h.typeTitle(this, type))
				return r;
		}

		"?";
	}
}
