/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include <kdb/extern.h>
#include <va_copy.h>

#include "kdb-priv.h"
#include "kdbfmt-priv.h"

#include <kfs/kfs-priv.h>
#include <kfs/directory.h>
#include <kfs/file.h>
#include <kfs/tar.h>
#include <kfs/arc.h>
#include <kfs/kfs-priv.h>
#include <klib/container.h>
#include <klib/text.h>
#include <klib/rc.h>
#include <sysalloc.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include <errno.h>


/*--------------------------------------------------------------------------
 * KDB utility
 */

/* KDBHdrValidate
 *  validates that a header sports a supported byte order
 *  and that the version is within range
 */
rc_t KDBHdrValidate ( const KDBHdr *hdr, size_t size,
    uint32_t min_vers, uint32_t max_vers )
{
    assert ( hdr != NULL );

    if ( size < sizeof * hdr )
        return RC ( rcDB, rcHeader, rcValidating, rcData, rcCorrupt );

    if ( hdr -> endian != eByteOrderTag )
    {
        if ( hdr -> endian == eByteOrderReverse )
            return RC ( rcDB, rcHeader, rcValidating, rcByteOrder, rcIncorrect );
        return RC ( rcDB, rcHeader, rcValidating, rcData, rcCorrupt );
    }

    if ( hdr -> version < min_vers || hdr -> version > max_vers )
        return RC ( rcDB, rcHeader, rcValidating, rcHeader, rcBadVersion );

    return 0;
}

/* KDBPathType
 *  checks type of path
 */
enum ScanBits
{
    scan_db     = ( 1 <<  0 ),
    scan_tbl    = ( 1 <<  1 ),
    scan_idx    = ( 1 <<  2 ),
    scan_col    = ( 1 <<  3 ),
    scan_idxN   = ( 1 <<  4 ),
    scan_data   = ( 1 <<  5 ),
    scan_dataN  = ( 1 <<  6 ),
    scan_md     = ( 1 <<  7 ),
    scan_cur    = ( 1 <<  8 ),
    scan_rNNN   = ( 1 <<  9 ),
    scan_lock   = ( 1 << 10 ),
    scan_odir   = ( 1 << 11 ),
    scan_ofile  = ( 1 << 12 ),
    scan_meta   = ( 1 << 13 ),
    scan_skey   = ( 1 << 14 ),
    scan_sealed = ( 1 << 15 )
};

static
rc_t CC scan_dbdir ( const KDirectory *dir, uint32_t type, const char *name, void *data )
{
    int *bits = data;

    type &= kptAlias - 1;

    if ( type == kptDir )
    {
        switch ( name [ 0 ] )
        {
        case 'c':
            if ( strcmp ( name, "col" ) == 0 )
            { * bits |= scan_col; return 0; }
            break;
        case 'm':
            if ( strcmp ( name, "md" ) == 0 )
            { * bits |= scan_md; return 0; }
            break;
        case 't':
            if ( strcmp ( name, "tbl" ) == 0 )
            { * bits |= scan_tbl; return 0; }
            break;
        case 'i':
            if ( strcmp ( name, "idx" ) == 0 )
            { * bits |= scan_idx; return 0; }
            break;
        case 'd':
            if ( strcmp ( name, "db" ) == 0 )
            { * bits |= scan_db; return 0; }
            break;
        }

        * bits |= scan_odir;
    }
    else if ( type == kptFile )
    {
        switch ( name [ 0 ] )
        {
        case 'l':
            if ( strcmp ( name, "lock" ) == 0 )
            { * bits |= scan_lock; return 0; }
            break;
        case 'i':
            if ( memcmp ( name, "idx", 3 ) == 0 )
            {
                if ( isdigit ( name [ 3 ] ) )
                { * bits |= scan_idxN; return 0; }
            }
            break;
        case 'd':
            if ( memcmp ( name, "data", 4 ) == 0 )
            {
                if ( name [ 4 ] == 0 )
                { * bits |= scan_data; return 0; }
                if ( isdigit ( name [ 4 ] ) )
                { * bits |= scan_dataN; return 0; }
            }
        case 'c':
            if ( strcmp ( name, "cur" ) == 0 )
            { * bits |= scan_cur; return 0; }
            break;
        case 'r':
            if ( isdigit ( name [ 1 ] ) && isdigit ( name [ 2 ] ) &&
                 isdigit ( name [ 3 ] ) && name [ 4 ] == 0 )
            { * bits |= scan_rNNN; return 0; }
            break;
        case 'm':
            if ( strcmp ( name, "meta" ) == 0 )
            { * bits |= scan_meta; return 0; }
            break;
        case 's':
            if ( strcmp ( name, "skey" ) == 0 )
            { * bits |= scan_skey; return 0; }
            if ( strcmp ( name, "sealed" ) == 0 )
            { * bits |= scan_sealed; return 0; }
            break;
        }

        * bits |= scan_ofile;
    }

    return 0;
}

int KDBPathType ( const KDirectory *dir, const char *path )
{
    const char *leaf, *parent;

    rc_t rc;
    int bits;
    int type = KDirectoryVPathType ( dir, path, NULL );
    switch ( type )
    {
    case kptDir:
    case kptDir | kptAlias:
        bits = 0;
        rc = KDirectoryVVisit ( dir, false, scan_dbdir, & bits, path, NULL );
        if ( rc == 0 )
        {
            /* look for a column */
            if ( ( bits & scan_idxN ) != 0 &&
                 ( bits & ( scan_data | scan_dataN ) ) != 0 )
            {
                if ( ( bits & ( scan_db | scan_tbl | scan_idx | scan_col ) ) == 0 )
                    type += kptColumn - kptDir;
                break;
            }

            /* look for a table */
            if ( ( bits & scan_col ) != 0 )
            {
                /* can't have sub-tables or a db */
                if ( ( bits & ( scan_db | scan_tbl ) ) == 0 )
                {
                    /* look for an old-structure table */
                    if ( ( bits & ( scan_meta | scan_md ) ) == scan_meta ||
                         ( bits & ( scan_skey | scan_idx ) ) == scan_skey )
                        type += kptPrereleaseTbl - kptDir;
                    else
                        type += kptTable - kptDir;
                }
                break;
            }

            /* look for metadata */
            if ( ( bits & ( scan_cur | scan_rNNN ) ) != 0 )
            {
                if ( ( bits & ( scan_db | scan_tbl | scan_idx | scan_col ) ) == 0 )
                    type += kptMetadata - kptDir;
                break;
            }

            /* look for a database */
            if ( ( bits & scan_tbl ) != 0 )
            {
                if ( ( bits & scan_col ) == 0 )
                    type += kptDatabase - kptDir;
                break;
            }

            /* look for a structured column */
            if ( ( bits & scan_odir ) != 0 )
            {
                leaf = strrchr ( path, '/' );
                if ( leaf != NULL )
                {
                    parent = string_rchr ( path, leaf - path, '/' );
                    if ( parent ++ == NULL )
                        parent = path;
                    if ( memcmp ( parent, "col/", 4 ) != 0 )
                        break;

                    bits = 0;
                    if ( KDirectoryVVisit ( dir, 1, scan_dbdir, & bits, path, NULL ) == 0 )
                    {
                        if ( ( bits & scan_idxN ) != 0 &&
                             ( bits & ( scan_data | scan_dataN ) ) != 0 )
                        {
                            if ( ( bits & ( scan_db | scan_tbl | scan_idx | scan_col ) ) == 0 )
                                type += kptColumn - kptDir;
                            break;
                        }
                    }
                }
            }
        }
        break;

    case kptFile:
    case kptFile | kptAlias:
    {
        /* if we hit a file first try it as an archive */
        rc_t rc;
        const KDirectory * ldir;

        rc = KDirectoryOpenSraArchiveRead_silent ( dir, &ldir, false, path );
        if ( rc != 0 )
            rc = KDirectoryOpenTarArchiveRead_silent ( dir, &ldir, false, path );

        /* it was an archive so recur */
        if (rc == 0)
        {
            /* recheck this newly opened directory for KDB/KFS type */
            int type2;

            type2 = KDBPathType ( ldir, "." );
            if ((type2 != kptDir) || (type != (kptDir|kptAlias)))
                type = type2;

            KDirectoryRelease (ldir);
        }
        /* it was not an archive so see if it it's an idx file */
        else
        {
            leaf = strrchr ( path, '/' );
            if ( leaf != NULL )
            {
                parent = string_rchr ( path, leaf - path, '/' );
                if ( parent ++ == NULL )
                    parent = path;
                if ( memcmp ( parent, "idx/", 4 ) == 0 )
                    type += kptIndex - kptFile;
            }
        }
        break;
    }
    }
    return type;
}


rc_t KDBOpenPathTypeRead ( const KDirectory * dir, const char * path, 
    const KDirectory ** pdir, int pathtype, int * ppathtype )
{
    rc_t rc;
    uint32_t dtype;
    int type;
    const KDirectory * ldir;

    /* set local directory pointer to error state NULL */
    ldir = NULL;
    rc = 0;

    type = dtype = KDirectoryPathType (dir, path);

    switch (dtype)
    {
    case kptDir:
    case kptDir | kptAlias:
        type = KDBPathType ( dir, path );

        if (( type == pathtype ) || (type == (pathtype|kptAlias)))
        {
            rc = KDirectoryVOpenDirUpdate ( ( KDirectory * ) dir, ( KDirectory ** ) & ldir, false, path, NULL );
            if ( rc != 0)
            {
                rc = KDirectoryVOpenDirRead ( dir, &ldir, false, path, NULL );
            }
        }
        break;

    case kptFile:
    case kptFile | kptAlias:
        rc = KDirectoryOpenSraArchiveRead_silent ( dir, &ldir, false, path );
        if ( rc != 0  )
            rc = KDirectoryOpenTarArchiveRead_silent ( dir, &ldir, false, path );
        if ( rc == 0 )
        {
            /* recheck this newly opened directory for KDB/KFS type */
            type = KDBPathType ( ldir, "." );

            /* it its not the KDB type we wanted fail */
            if (( type != pathtype ) && (type != (pathtype | kptAlias )))
            {
                KDirectoryRelease (ldir);
                ldir = NULL;
                rc = RC ( rcDB, rcMgr, rcOpening, rcPath, rcIncorrect );
            }
        }
        else if ( rc != 0 )
        {
            rc = RC ( rcDB, rcMgr, rcOpening, rcPath, rcIncorrect );
        }
        break;
    }

    if (( rc != 0 ) || (( type != pathtype ) && (type != (pathtype|kptAlias))))
    {
        /* tune the error message based on path type */
        uint32_t obj;
        switch ( pathtype )
        {
        default:
            obj = rcType;
            break;
        case kptTable:
        case kptTable | kptAlias:
        case kptPrereleaseTbl:
        case kptPrereleaseTbl | kptAlias:
            obj = rcTable;
            break;
        case kptColumn:
        case kptColumn | kptAlias:
            obj = rcColumn;
            break;
        case kptDatabase:
        case kptDatabase | kptAlias:
            obj = rcDatabase;
            break;
        }
        switch ( type )
        {
        case kptNotFound:
            rc =  RC ( rcDB, rcMgr, rcOpening, obj, rcNotFound );
            break;
        case kptBadPath:
            rc =  RC ( rcDB, rcMgr, rcOpening, rcPath, rcInvalid );
            break;
        default:
            rc = RC ( rcDB, rcMgr, rcOpening, rcPath, rcIncorrect );
            break;
        }
    }
    if ((rc != 0) || ((ldir != NULL) && (pdir == NULL)))
    {
        KDirectoryRelease ( ldir );
        ldir = NULL;
    }

    if (pdir != NULL)
        *pdir = ldir;

    return rc;
}


/* Writable
 *  examines a directory structure for any "lock" files
 */
rc_t KDBWritable ( const KDirectory *dir, const char *path )
{
    uint32_t access;
    rc_t rc;

    /* we have to be able to check the access if it is to be writable */
    rc = KDirectoryVAccess ( dir, & access, path, NULL );
    if ( rc == 0 )
    {
        /* if there is a lock (or deprecated sealed) file in this directory */
        switch ( KDirectoryPathType ( dir, "%s/lock", path ) )
        {
        case kptFile:
        case kptFile | kptAlias:
            rc = RC ( rcDB, rcPath, rcAccessing, rcLock, rcLocked );
            break;
        case kptNotFound:
            /* much simpler handling for the sealed file */
            switch ( KDirectoryPathType ( dir, "%s/sealed", path ) )
            {
            case kptFile:
            case kptFile | kptAlias:
                rc = RC ( rcDB, rcPath, rcAccessing, rcLock, rcLocked );
                break;
            case kptNotFound:
                /* check if there are no write permissions */
                if ( ( access & 0222 ) == 0 )
                    rc = RC ( rcDB, rcPath, rcAccessing, rcPath, rcReadonly );
                /* else rc is still 0 from VAccess */
            }
            break;
        case kptBadPath:
            /* likely to be a non-driectory or something */
            rc = RC ( rcDB, rcPath, rcAccessing, rcPath, rcInvalid);
            break;
        default:
            /* an illegal type of object named "lock" is in this directory
             * which will block the ability to lock it
             */
            rc = RC ( rcDB, rcPath, rcAccessing, rcPath, rcIncorrect );
        }
    }

    return rc;
}


LIB_EXPORT bool CC KDBIsLocked ( const KDirectory *dir, const char *path )
{
    return ( KDBWritable (dir, path) != 0 );
}


/* GetObjModDate
 *  extract mod date from a path
 */
rc_t KDBGetObjModDate ( const KDirectory *dir, KTime_t *mtime )
{
    /* HACK ALERT - there needs to be a proper way to record modification times */
    
    /* this only tells the last time the table was locked,
       which may be close to the last time it was modified */
    rc_t rc = KDirectoryDate ( dir, mtime, "lock" );
    if ( rc == 0 )
        return 0;

    if ( GetRCState ( rc ) == rcNotFound )
    {
        rc = KDirectoryDate ( dir, mtime, "sealed" );
        if ( rc == 0 )
            return 0;
    }

    /* get directory timestamp */
    rc = KDirectoryDate ( dir, mtime, "." );
    if ( rc == 0 )
        return 0;

    * mtime = 0;
    return rc;
}

/* GetPathModDate
 *  extract mod date from a path
 */
rc_t KDBVGetPathModDate ( const KDirectory *dir,
    KTime_t *mtime, const char *path, va_list args )
{
    rc_t rc;
    uint32_t ptype;
    const KDirectory *obj_dir;

    va_list cpy;
    va_copy ( cpy, args );
    ptype = KDirectoryVPathType ( dir, path, cpy );
    va_end ( cpy );

    switch ( ptype )
    {
    case kptDir:
    case kptDir | kptAlias:
        break;

    default:
        return KDirectoryVDate ( dir, mtime, path, args );
    }

    * mtime = 0;
    rc = KDirectoryVOpenDirRead ( dir, & obj_dir, true, path, args );
    if ( rc == 0 )
    {
        rc = KDBGetObjModDate ( obj_dir, mtime );
        KDirectoryRelease ( obj_dir );
    }

    return rc;
}


/* KDBVMakeSubPath
 *  adds a namespace to path spec
 */
rc_t KDBVMakeSubPath ( struct KDirectory const *dir,
    char *subpath, size_t subpath_max, const char *ns,
    uint32_t ns_size, const char *path, va_list args )
{
    rc_t rc;

    if ( ns_size > 0 )
    {
        subpath += ns_size + 1;
        subpath_max -= ns_size + 1;
    }

    /* because this call only builds a path instead of resolving anything
     * is is okay that we are using the wrong directory */
    rc = KDirectoryVResolvePath ( dir, false,
        subpath, subpath_max, path, args );
    switch ( GetRCState ( rc ) )
    {
    case 0:
        assert ( subpath [ 0 ] != 0 );
        if ( subpath [ 0 ] == '.' || subpath [ 1 ] == '/' )
            return RC ( rcDB, rcDirectory, rcResolving, rcPath, rcInvalid );
        break;
    case rcInsufficient:
        return RC ( rcDB, rcDirectory, rcResolving, rcPath, rcExcessive );
    default:
        return rc;
    }

    if ( ns_size != 0 )
    {
        subpath -= ns_size + 1;
        memcpy ( subpath, ns, ns_size );
        subpath [ ns_size ] = '/';
    }
    return rc;
}
