/****************************************************************************
    
    dssi_shm.h - functions that can handle shared memory segments for
                 DSSI plugins and UIs
    
    Copyright (C) 2005  Lars Luthman <larsl@users.sourceforge.net>
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 01222-1307  USA

****************************************************************************/

#ifndef DSSI_SHM_H
#define DSSI_SHM_H

/** @file
    This file contains functions for handling shared memory in DSSI UIs and
    plugins. It is used by the DSSIUIClient class on the UI side, but can also
    be used on its own. 
*/

#if defined(__cplusplus)
extern "C" {
#endif
  
  /** This function should be used in the UI to allocate a segment of shared
      memory and generate a key for this segment that can be sent to the 
      plugin. @c bytes is the requested segment size in bytes. @c key and
      @c used_flag are output parameters, @c key will hold a key string for
      the new segment that you can send to the plugin with a /configure call,
      and @c used_flag will contain a pointer to a char in the shared memory
      segment that the plugin will set to 1 when it has attached succesfully
      to the segment. You will have to deallocate @c *key yourself.
      @returns A pointer to the new segment or NULL
  */
  void* dssi_shm_allocate(size_t bytes, char** key, char** used_flag);
  
  /** This function should be used by the plugin to attach to a shared memory
      segment that has been allocated by the UI using dssi_shm_allocate().
      @c key is the key string received from the UI and @c old_ptr is a
      pointer to the old shared memory segment if this is not the first time
      the plugin tries to attach to a segment. It can be NULL if no segment
      has been attached earlier. 
      @returns A pointer to the new attached segment or NULL
  */
  void* dssi_shm_attach(const char* key, void* old_ptr);
  
  /** This should always be called by the UI when it wants to deallocate
      a shared memory segment allocated using dssi_shm_allocate(). @c key
      is the key string generated by the call to dssi_shm_allocate().
      @returns The result of a @c shmctl() call
  */
  int dssi_shm_free(const char* key);
  
  /** This should always be called by the plugin when it wants to detach
      from a memory segment that has been attached using dssi_shm_attach().
      @c ptr should be a pointer to the memory segment.
      @returns The result of s @c shmdt() call
  */
  int dssi_shm_detach(void* ptr);
  
#if defined(__cplusplus)
}
#endif

#endif
