% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/case-when.R
\name{vec-case-and-replace}
\alias{vec-case-and-replace}
\alias{vec_case_when}
\alias{vec_replace_when}
\title{Recode and replace using logical conditions}
\usage{
vec_case_when(
  conditions,
  values,
  ...,
  default = NULL,
  unmatched = "default",
  ptype = NULL,
  size = NULL,
  conditions_arg = "conditions",
  values_arg = "values",
  default_arg = "default",
  error_call = current_env()
)

vec_replace_when(
  x,
  conditions,
  values,
  ...,
  x_arg = "x",
  conditions_arg = "conditions",
  values_arg = "values",
  error_call = current_env()
)
}
\arguments{
\item{conditions}{A list of logical condition vectors.

For \code{vec_case_when()}, each vector should be the same size.

For \code{vec_replace_when()}, each vector should be the same size as \code{x}.

Where a value in \code{conditions} is \code{TRUE}, the corresponding value in
\code{values} will be assigned to the result.}

\item{values}{A list of vectors.

For \code{vec_case_when()}, each vector should be size 1 or the size implied by
\code{conditions}. The common type of \code{values} and \code{default} determine the
output type, unless overridden by \code{ptype}.

For \code{vec_replace_when()}, each vector should be size 1 or the same size
as \code{x}. Each vector will be cast to the type of \code{x}.}

\item{...}{These dots are for future extensions and must be empty.}

\item{default}{Default value to use when \code{conditions} does not match every
location in the output.

By default, a missing value is used as the default value.

If supplied, \code{default} must be size 1 or the size implied by \code{conditions}.

Can only be set when \code{unmatched = "default"}.}

\item{unmatched}{Handling of unmatched locations.

One of:
\itemize{
\item \code{"default"} to use \code{default} in unmatched locations.
\item \code{"error"} to error when there are unmatched locations.
}}

\item{ptype}{An optional override for the output type, which is usually
computed as the common type of \code{values} and \code{default}.}

\item{size}{An optional override for the output size, which is usually
computed as the size of the first element of \code{conditions}.

Only useful for requiring a fixed size when \code{conditions} is an empty list.}

\item{error_call}{The execution environment of a currently
running function, e.g. \code{caller_env()}. The function will be
mentioned in error messages as the source of the error. See the
\code{call} argument of \code{\link[rlang:abort]{abort()}} for more information.}

\item{x}{A vector.}

\item{x_arg, conditions_arg, values_arg, default_arg}{Argument names used in
error messages.}
}
\value{
A vector.
\itemize{
\item For \code{vec_case_when()}, the type of the output is computed as the common
type of \code{values} and \code{default}, unless overridden by \code{ptype}. The names of
the output come from the names of \code{values} and \code{default}. The size of the
output comes from the implied size from \code{conditions}, unless overridden by
\code{size}.
\item For \code{vec_replace_when()}, the type of the output will have the same type as
\code{x}. The names of the output will be the same as the names of \code{x}. The size
of the output will be the same size as \code{x}.
}
}
\description{
\itemize{
\item \code{vec_case_when()} constructs an entirely new vector by recoding the \code{TRUE}
\code{conditions} to their corresponding \code{values}. If there are locations not
matched by \code{conditions}, then they are recoded to the \code{default} value.
\item \code{vec_replace_when()} updates an existing vector by replacing the values
from \code{x} matched by the \code{TRUE} \code{conditions} with their corresponding
\code{values}. In this case, each element of \code{values} must have the same type as
\code{x} and locations not matched by \code{conditions} retain their original \code{x}
value.
}

\code{vec_case_when()} is often thought of as a way to vectorize multiple if-else
statements, and is an R equivalent of the SQL "searched" \verb{CASE WHEN}
statement.
}
\examples{
# Note how the first `TRUE` is used in the output.
# Also note how the `NA` falls through to `default`.
x <- seq(-2L, 2L, by = 1L)
x <- c(x, NA)
conditions <- list(
  x < 0,
  x < 1
)
values <- list(
  "<0",
  "<1"
)
vec_case_when(
  conditions,
  values,
  default = "other"
)

# Missing values need to be handled with their own case
# if you want them to have a special value
conditions <- list(
  x < 0,
  x < 1,
  is.na(x)
)
values <- list(
  "<0",
  "<1",
  NA
)
vec_case_when(
  conditions,
  values,
  default = "other"
)

# Both `values` and `default` are vectorized
values <- list(
  x * 5,
  x * 10,
  NA
)
vec_case_when(
  conditions,
  values,
  default = x * 100
)

# Use `vec_replace_when()` if you need to update `x`, retaining
# all previous values in locations that you don't match
conditions <- list(
  x < 0,
  x < 1
)
values <- list(
  0,
  1
)
out <- vec_replace_when(
  x,
  conditions,
  values
)
out

# Note how `vec_replace_when()` is type stable on `x`, we retain the
# integer type here even though `values` contained doubles
typeof(out)

# `vec_case_when()` creates a new vector, so names come from `values`
# and `default`. `vec_replace_when()` modifies an existing vector, so
# names come from `x` no matter what, just like `[<-` and `base::replace()`
x <- c(a = 1, b = 2, c = 3)
conditions <- list(x == 1, x == 2)
values <- list(c(x = 0), c(y = -1))
vec_case_when(conditions, values)
vec_replace_when(x, conditions, values)

# If you want to enforce that you've covered all of the locations in your
# `conditions`, use `unmatched = "error"` rather than providing a `default`
x <- c(0, 1, 2)
conditions <- list(x == 1, x == 2)
values <- list("a", "b")
try(vec_case_when(conditions, values, unmatched = "error"))
}
