#!/bin/bash
#
# process.sh
# Shows processes using the Nvidia GPU
#
# Copyright (c) 2020 Rui Guilherme da Cunha
# Copyright (c) 2021 Erich Eickmeyer
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

_nvidiaReport="$(nvidia-smi -q --xml-format | grep -oP '(?<=<pid>).*(?=</pid>)|(?<=<process_name>).*(?=--type)|(?<=<process_name>).*(?=</process_name>)|(?<=<used_memory>).*(?=</used_memory>)')"
_valueIdx=1
_procList=()
_zenityExeStr=$( command -v 'zenity' );

# Bail if zenity is not usable (e.g. bad environment, broken deps)
if [ -z "${_zenityExeStr}" ] || [ ! -x "${_zenityExeStr}" ]; then
  1>&2 echo 'zenity is not executable. 
Please ensure it is installed and in your PATH.';
  exit 1;
fi

# Disable shellcheck search/replace expansion warning
# shellcheck disable=SC2001
for _valueStr in $(echo "${_nvidiaReport}" | sed -e 's/ //g')
do
  case "${_valueIdx}" in
    1) _pid="${_valueStr}";;
    2) _process="${_valueStr}";;
    3) _memory="${_valueStr}";;
  esac

  ((_valueIdx++))
  if [ "${_valueIdx}" -gt 3 ]; then
    _valueIdx=1;
    _procList=("${_procList[@]}" "${_pid}" "${_process}" "${_memory}")
  fi
done

_selectedPid="$("${_zenityExeStr}" --width=600 --height=400 \
  --list \
  --separator="$\n" \
  --text="Running Processes:" \
  --ok-label="Close/Restart Selected Process" \
  --cancel-label="Ok" \
  --column "PID" \
  --column "Process" \
  --column "Memory" \
  "${_procList[@]}")";

if [ -n "${_selectedPid}" ]; then
  _killId=1;
  _reloadId=2;
  _selectedAction="$("${_zenityExeStr}" --width=600 --height=400 \
  --list \
  --separator="$\n" \
  --text="Select one option. PID: ${_selectedPid}" \
  --column "Exec" \
  --column "Description" \
  ${_killId} "Kill process (this can cause issue in the application)" \
  ${_reloadId} "Reload process")";
  if [ "${_selectedAction}" = "${_killId}" ]; then
    kill -9 "${_selectedPid}";
    exit 0;
  elif [ "${_selectedAction}" = "${_reloadId}" ]; then
    kill -HUP "${_selectedPid}";
    exit 0;
  else
    exit 0;
  fi
else
  exit 0;
fi
