/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef _GTLCORE_LEXER_BASE_H_
#define _GTLCORE_LEXER_BASE_H_

#include <sstream>
#include <GTLCore/String.h>

#define IDENTIFIER_IS_KEYWORD( tokenname, tokenid) \
  if( identifierStr == tokenname) \
  { \
    return GTLCore::Token(GTLCore::Token::tokenid, line(), initial_col); \
  }

#define CHAR_IS_TOKEN( tokenchar, tokenid ) \
  if( lastChar == tokenchar ) \
  { \
    return GTLCore::Token(GTLCore::Token::tokenid, line(), initial_col); \
  }

#define CHAR_IS_TOKEN_OR_TOKEN( tokenchar, tokendecidechar, tokenid_1, tokenid_2 ) \
  if( lastChar == tokenchar  ) \
  { \
    if( getNextChar() == tokendecidechar ) \
    { \
      return GTLCore::Token(GTLCore::Token::tokenid_2, line(), initial_col); \
    } else { \
      unget(); \
      return GTLCore::Token(GTLCore::Token::tokenid_1, line(), initial_col); \
    } \
  }

#define CHAR_IS_TOKEN_OR_TOKEN_OR_TOKEN( tokenchar_1, tokenchar_2, tokenchar_3, tokenid_1, tokenid_2, tokenid_3 ) \
  if( lastChar == tokenchar_1 ) \
  { \
    int nextChar = getNextChar(); \
    if( nextChar == tokenchar_2 ) \
    { \
      return GTLCore::Token(GTLCore::Token::tokenid_2, line(), initial_col); \
    } else if( nextChar  == tokenchar_3 ) \
    { \
      return GTLCore::Token(GTLCore::Token::tokenid_3, line(), initial_col); \
    } else { \
      unget(); \
      return GTLCore::Token(GTLCore::Token::tokenid_1, line(), initial_col); \
    } \
  }

namespace GTLCore {
  class Token;
  /**
   * @internal
   * This class provide some base functions to lexers. Such as eating spaces,
   * or managing the flow.
   * @ingroup GTLCore
   */
  class GTLCORE_EXPORT LexerBase {
    public:
      LexerBase(std::istream* sstream);
      virtual ~LexerBase();
    public:
      virtual GTLCore::Token nextToken() =0;
    protected:
      /**
       * @return the next char and increment the column counter.
       */
      int getNextChar();
      /**
       * @return the next char that is not a separator (space, tab, return...)
       */
      int getNextNonSeparatorChar();
      /**
       * Cancel the reading of the previous char.
       */
      void unget();
      bool eof() const;
      int line() const;
      int column() const;
      /**
       * Call this function to ignore C++ style comments.
       * @return true if there was a comment, and that _token got fill with
       *         the token after the comment, false other wise (in which case
       *         _token content is undefined)
       */
      bool ignoreComment(GTLCore::Token& _token, int _lastChar  );
      /**
       * Get an identifier (or keyword) in the current flow of character.
       */
      GTLCore::String getIdentifier(int lastChar);
      GTLCore::Token getDigit(int lastChar);
      GTLCore::Token getString(int lastChar);
    private:
      struct Private;
      Private* const d;
  };
}

#endif
