#ifndef _NEARMASKE_H
#define _NEARMASKE_H
#include "osl/state/numEffectState.h"
#include "osl/misc/bitOp.h"
#include "osl/ptypeTraits.h"
#include <iosfwd>

namespace osl
{
  namespace container
  {
  /**
   * あるマスの10近傍(8近傍+桂馬の利き)の駒の有無を表すbitset.
   * 手番の側からの固定視点
   * 0-9 bit - Direction(UL=0, .., UUR=9)に対応するマスが空白または敵駒の時に1
   * 16-25 bit - Direction+16(UL=16, .., UUR=25)に対応するマスが空白の時に1
   * 先手番で，マス(3,3)に注目している時に，ULに対応するマスは，(4,2)ではなく
   * (2,4)を指す．(2,4)にULの利きを持つ味方の駒を置くと(3,3)を攻撃できるから
   */
  class NearMask
  {
    unsigned int mask;
    explicit NearMask(unsigned int value) : mask(value) {}
  public:
    NearMask() : mask(0) {}
    static const NearMask makeDirect(unsigned int value)
    {
      return NearMask(value);
    }
    /**
     * あるマスに,利きをつけようとする場合に, 10近傍の状態を返す
     * @param P(template) - 攻撃側のプレイヤー
     * @param Dir(template) - この向きの利きをつける場合のマス
     * @param state - 盤面を表す
     * @param target - 利きをつけられるマス
     * @return 下位16ビット(中の10ビット)は空白または敵駒の時に1,それ以外で0
     * 上位16ビット(中の10ビット) は空白の時に1, それ以外で0
     */
    template<Player P,Direction Dir>
    static const NearMask make(NumEffectState const& state,Square target) {
      /** 攻撃する側のマス */
      Square pos=target-DirectionPlayerTraits<Dir,P>::offset();
      Piece p=state.pieceAt(pos);
      if(p.canMoveOn<P>()){
	if(p.isEmpty()) 
	  return NearMask(0x10001<<static_cast<int>(Dir));
	return NearMask(1<<static_cast<int>(Dir));
      }
      return NearMask(0);
    }
    template<Player P>
    static const NearMask make(NumEffectState const& state,Square target) {
      NearMask nearMask;
      /**
       * 10近傍の駒の状態をチェック
       * NOTE: Stateがsignature effectの時には特殊化した高速バージョンを
       * 作成すべし
       */
      nearMask=NearMask::make<P,U>(state,target);
      nearMask|=NearMask::make<P,UL>(state,target);
      nearMask|=NearMask::make<P,UR>(state,target);
      nearMask|=NearMask::make<P,L>(state,target);
      nearMask|=NearMask::make<P,R>(state,target);
      nearMask|=NearMask::make<P,DL>(state,target);
      nearMask|=NearMask::make<P,D>(state,target);
      nearMask|=NearMask::make<P,DR>(state,target);
      nearMask|=NearMask::make<P,UUL>(state,target);
      nearMask|=NearMask::make<P,UUR>(state,target);
      return nearMask;
    }

    NearMask& operator|=(NearMask r) { mask |= r.mask; return *this; }
    NearMask& operator&=(NearMask r) { mask &= r.mask; return *this; }

    unsigned int spaceMask() const { return mask >> 16; }
    bool isSet(int n) const
    {
      return mask & (1<<n);
    }
    /**
     * on の bit を一つoffにし，そのindexを返す
     */
    int takeOneBit()
    {
      return misc::BitOp::takeOneBit(mask);
    }
    bool any() const { return mask; }
    unsigned int uintValue() const { return mask; }
  };

  inline const NearMask operator|(NearMask l, NearMask r) { 
    NearMask result = l;
    return result |= r;
  }
  inline const NearMask operator&(NearMask l, NearMask r) { 
    NearMask result = l;
    return result &= r;
  }
  inline bool operator==(NearMask l, NearMask r)
  {
    return l.uintValue() == r.uintValue();
  }
  inline bool operator!=(NearMask l, NearMask r)
  {
    return ! (l == r);
  }
  inline bool operator<(NearMask l, NearMask r)
  {
    return l.uintValue() < r.uintValue();
  }

  std::ostream& operator<<(std::ostream&, NearMask);
} // namespace container
  using container::NearMask;
} // namespace osl
#endif /* _NEARMASK_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
