/*
 * Kylin-video
 *
 * Copyright (C) 2021, Tianjin KYLIN Information Technology Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Liu Cong <liucong1@kylinos.cn>
 *
 */

#ifndef PLAYLISTWIDGET_H
#define PLAYLISTWIDGET_H

#include <QWidget>
#include <QMap>

#include <mutex>
#include "playlist.h"
#include "global/globalsignal.h"

#define FONT_COLOR_SELECTED "rgb(55,144,250)"
#define AnmationDuration 300

class PlayItem;
class PlayList;
class PlayListItemMenu;

class QLabel;
class ImageLabel;
class QListWidget;
class QVBoxLayout;
class QHBoxLayout;
class QPushButton;
class ListLoopMenu;
class QListWidgetItem;
class QVariantAnimation;
class QPropertyAnimation;

/// 列表模式-预览模式
enum ListViewMode{LIST, VIEW};

class MarkListItem : public QWidget
{
    Q_OBJECT
public:
    explicit MarkListItem(QWidget *parent = nullptr);
    ~MarkListItem();

    int getPos(){return m_markPos;}
    void setData(QString path, int duration, int pos, QString desc, QString preview = "");
    void setMode(ListViewMode mode);
    void setEnter();
    void setLeave();
    QString getFilePath() {return m_tooltip;}
    void setWidgetFont(QFont f);

public slots:
    void setNotExit(QString);
    void setFontColor(QString);
    void setBlackTheme();
    void setLightTheme();

signals:
    void sigPlayAndSeek(QString, int);
    void sigRightBtnPressed(int);
    void sigDeleteMark(QString, int);

private:
    void initStyle();
    void updateNameShow();

private:
    int m_markPos;
    bool m_needSeekToPos,
         m_fileExit;
    QString m_currentPlayingFile;
    QString m_fileName;
    QString m_tooltip;
    QString m_describe;
    QSize m_noPreviewSize, m_previewSize;
    QPixmap *m_pixmapPreview;
    ImageLabel  *m_labPreview;
    QLabel  *m_labFilePath,
            *m_labDuration,
            *m_labMarkPos;
    QWidget *m_dp;

    QVBoxLayout *m_vlay;
    QHBoxLayout *m_hlay;

    QPushButton *m_btnDelete;

    ListViewMode m_mode;

protected:
    void mousePressEvent(QMouseEvent *e) override;
    void mouseDoubleClickEvent(QMouseEvent *e) override;
    void enterEvent(QEvent *e) override;
    void leaveEvent(QEvent *e) override;
};

/// 播放列表条目
class PlayListItem : public QWidget
{
    Q_OBJECT
public:
    explicit PlayListItem(QWidget *parent = nullptr);
    ~PlayListItem();

    QString getPath() {return m_tip;}
    QString getName() {return m_showName;}
    void setSelected(bool s);
    void setData(QString _file, int _duration, int _lastTime, QString _viewPath = "");
    void setDuration(int duration);
    void setEnter();
    void setLeave();
    void toBig();
    void toSmall();
    void reloadView();
    void setMode(ListViewMode _m_mode);
    bool getExist(){return m_isExist;}

    void setBlackTheme();
    void setLightTheme();
    void setWidgetFont(const QFont &f);

signals:
    void sigValChange(int);
    void sigRightBPressed(QString);
    void sigNotExists(QString);
    void sigExistsStateChange(QString, bool);

public slots:
    void slotNotExit(QString);

private:
    QLabel  *m_labIcon,
            *m_labSpacing,
            *m_labName,
            *m_labDuration;
    QPixmap *m_pixmapPreview;
    ImageLabel  *m_labPreview;
    QPushButton *m_btnDelete;

    QVariantAnimation   *m_anmToBig,
                        *m_anmToSmall;

    QString m_tip,
            m_showName,
            m_viewPath;
    bool    m_isSelected,
            m_isExist,
            m_isMark;
    ListViewMode m_mode;

    void initDeleteButton();
    void initAnimation();
    void updateNameShow();
    void updateText();

private slots:
    void setFontColor(QString);
    void deleteItem();

protected:
    void mousePressEvent(QMouseEvent *e) override;
    void mouseDoubleClickEvent(QMouseEvent *e) override;
};

namespace Ui {
class PlayListWidget;
}

class PlayListWidget : public QWidget
{
    Q_OBJECT

public:
    explicit PlayListWidget(QWidget *parent = nullptr);
    ~PlayListWidget();

    bool isShow(){return m_isShow;}
    PlayList *getPlayList(){return m_playList;}

signals:
    void sigAddItem(QString, int, int, QString, bool);
    void sigMove(int);

public slots:
    void showItemMenu(QString file);
    void addItem(PlayItem _item);
    void deleteItem(QString file);
    void clearList();

    void updateMarks(QVector<MarkItem> markvec);
    void deleteMark(QString filepath, int pos);
    void clearMark();

    void reloadView(QString file);
    void resetDuration(QString file, int duration);
    void slotShow();
    void slotHide();

    // 展开按钮更新
    void setShowButton(bool show);
    void updateShowIcon();
    void updateHideIcon();

    void delAvailableIndex(QString _file);
    void changeListViewMode();
    void setPlayIndex(int _index);
    void selectNone();

    void setBlackTheme();
    void setLightTheme();

    void gSettingChanged(QString key);

private:
    Ui::PlayListWidget *ui;

    QString m_theme;
    QString m_currentPlayingFile;
    QString m_selectedFile;
    PlayList *m_playList;
    QListWidget *m_playListWidget,
                *m_markListWidget;

    PlayOrder m_playOrder;          // 播放顺序
    PlayListItemMenu *m_itemMenu;   // 右击列表选项菜单
    QMap<QString, QListWidgetItem*> m_itemMap;  // 列表item map，key:绝对路径

    std::mutex m_mux;
    std::mutex m_itemAddMux;
    bool m_isShow;
    bool m_isMouseEnter;
    ListViewMode m_mode;
    QPropertyAnimation *m_showAnm;
    QPropertyAnimation *m_hideAnm;

    void initMenu();
    void initStackedWidget();
    void initPlayList();
    void initGlobalSig();
    void initAnimation();
    void initConnection();
    void initDefaultWidget();

    void updateOrderIcon();
    void updateViewIcon();
    void updateAddIcon();
    void updateDeleteIcon();

protected:
    bool eventFilter(QObject *watched, QEvent *event) override;
    void enterEvent(QEvent *e) override;
    void leaveEvent(QEvent *e) override;
    void moveEvent(QMoveEvent *event) override;
};

#endif // PLAYLISTWIDGET_H
