/*-------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation.  All rights reserved.
 *
 * src/configuration/certs.rs
 *
 *-------------------------------------------------------------------------
 */

use serde::Deserialize;

/// Specifies the type of certificate input configuration.
///
/// This enum determines how certificates are provided to the gateway:
/// - `PemFile`: Use existing PEM certificate and key files
/// - `PemAutoGenerated`: Auto-generate self-signed certificates if they don't exist
#[derive(Debug, Deserialize, Default, Clone, PartialEq, Eq)]
#[serde(rename_all = "PascalCase")]
pub enum CertInputType {
    /// Use existing PEM certificate and private key files
    #[default]
    PemFile,
    /// Auto-generate self-signed certificates in PEM format if they don't exist
    PemAutoGenerated,
}

/// Configuration options for certificate handling.
///
/// This struct contains the paths and type information needed to configure
/// SSL/TLS certificates for the gateway server.
#[derive(Debug, Deserialize, Default, Clone)]
#[serde(rename_all = "PascalCase")]
pub struct CertificateOptions {
    /// The type of certificate input (PEM files or auto-generation)
    pub cert_type: CertInputType,
    /// Path to the certificate file (required for PemFile type)
    pub file_path: Option<String>,
    /// Path to the private key file (required for PemFile type)
    pub key_file_path: Option<String>,
    /// Path to the CA certificate chain file (optional)
    pub ca_path: Option<String>,
}
