/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import fs from 'fs';
import url from 'url';
import path from 'path';
import https from 'https';

const currentFilePath = url.fileURLToPath(import.meta.url);
const buildFilePath = path.join(path.dirname(currentFilePath), 'build-playwright-driver.sh');
const latestNodeJsLts = await getLatestNodeJsVersion();

refreshBuildScriptVersionNumber(latestNodeJsLts, buildFilePath);
console.log(`Updated build script ${path.relative(process.cwd(), buildFilePath)} to Node.js ${latestNodeJsLts}`);

function getLatestNodeJsVersion() {
  return new Promise((resolve, reject) => {
    https.get('https://nodejs.org/dist/index.json', res => {
      let content = '';
      res.on('data', chunk => content += chunk);
      res.on('end', () => {
        if (res.statusCode !== 200)
          reject(new Error(`Failed to get nodejs version, status code: ${res.statusCode}`));
        const version = JSON.parse(content);
        const latestLts = version.find(release => release.lts);
        resolve(latestLts.version.slice(1));
      });
    });
  });
}

function refreshBuildScriptVersionNumber(newNodeJsVersion, buildScriptPath) {
  const replacementRegex = new RegExp(`(NODE_VERSION=")(.*)(" # autogenerated via \\.\/${path.basename(currentFilePath).replace('.', '\\.')})`);
  let buildFile = fs.readFileSync(buildFilePath, 'utf8');
  buildFile = buildFile.replace(replacementRegex, `$1${latestNodeJsLts}$3`);
  fs.writeFileSync(buildFilePath, buildFile);
}
