/* ----- encode/653x1541, derived from supercop/crypto_encode/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *encode_653x1541_checksums[] = {
  "1b20d01f7809f5516f698fdfb9bcf69a9b077a03bc594f94f427983eed219924",
  "e9661445e5dc394398e8c9fc031cde1c405c4e325fce3297c6e8bdb53310aa70",
} ;

static void (*crypto_encode)(unsigned char *,const void *);
#define crypto_encode_STRBYTES ntruprime_encode_653x1541_STRBYTES
#define crypto_encode_ITEMS ntruprime_encode_653x1541_ITEMS
#define crypto_encode_ITEMBYTES ntruprime_encode_653x1541_ITEMBYTES

static void *storage_encode_653x1541_s;
static unsigned char *test_encode_653x1541_s;
static void *storage_encode_653x1541_x;
static unsigned char *test_encode_653x1541_x;
static void *storage_encode_653x1541_s2;
static unsigned char *test_encode_653x1541_s2;
static void *storage_encode_653x1541_x2;
static unsigned char *test_encode_653x1541_x2;

#define precomputed_encode_653x1541_NUM 4

static const unsigned char precomputed_encode_653x1541_s[precomputed_encode_653x1541_NUM][crypto_encode_STRBYTES] = {
  {149,195,1,228,193,163,144,202,166,89,25,95,67,77,255,97,121,61,183,220,136,37,54,82,180,191,124,73,163,67,14,216,147,226,235,250,89,213,106,178,204,225,133,31,118,106,174,124,97,114,55,0,178,160,170,164,234,207,121,39,123,180,57,51,186,14,152,228,53,222,82,15,86,215,81,8,1,110,144,142,141,242,100,65,204,197,244,111,80,178,140,36,236,142,200,64,146,0,255,91,252,220,184,51,61,226,14,40,230,2,150,64,101,111,92,22,210,177,130,197,155,119,77,116,3,88,234,197,33,169,227,192,160,54,152,55,70,198,115,109,243,150,68,166,122,9,193,132,140,209,183,225,71,123,205,146,213,146,126,165,255,151,17,141,168,128,26,70,223,213,139,58,197,71,181,34,15,38,44,42,140,112,124,23,108,117,54,150,8,182,36,114,225,189,244,36,222,1,77,194,157,161,132,11,244,110,82,141,132,215,1,42,42,129,177,224,228,143,125,220,212,88,152,119,134,1,254,159,214,0,205,113,24,22,167,24,76,66,121,70,173,127,16,58,40,64,207,80,187,33,239,109,56,146,20,105,137,9,126,126,22,59,21,18,117,54,167,156,137,5,189,233,70,147,135,151,145,200,172,32,208,16,6,107,250,204,24,119,201,243,5,202,113,136,63,142,93,139,32,253,79,100,90,131,225,245,228,205,255,109,91,125,233,19,133,86,91,84,22,41,74,236,12,3,27,20,135,128,231,153,118,208,202,60,215,42,7,162,1,239,153,65,151,70,213,235,243,91,244,154,206,122,14,152,204,35,60,178,131,233,116,22,121,9,92,24,234,66,195,181,71,38,120,14,130,135,85,221,136,161,167,20,205,236,204,86,171,20,12,114,253,147,250,18,190,105,221,82,231,227,205,136,18,204,40,126,177,10,14,248,221,153,78,114,137,243,216,16,255,163,222,0,217,94,200,193,58,227,60,157,104,247,204,28,120,175,239,160,131,142,88,138,217,31,251,107,35,113,206,113,192,251,33,82,40,50,236,225,85,174,179,35,233,197,245,10,203,81,32,105,207,30,174,4,82,214,204,46,221,98,66,118,154,206,154,35,239,162,120,28,123,240,27,24,47,177,206,191,212,20,182,248,235,175,146,121,114,15,65,56,169,254,27,78,92,58,50,226,83,81,168,43,70,10,6,221,21,117,7,12,249,13,71,223,212,235,242,144,72,49,114,126,155,138,14,213,201,252,136,205,54,215,65,252,127,95,83,171,200,222,196,221,91,72,171,76,83,205,157,101,136,10,126,83,247,42,102,53,124,202,178,212,92,92,79,123,84,163,86,48,164,74,138,78,72,203,86,141,242,171,25,21,15,77,111,119,219,52,237,213,179,148,189,254,75,111,108,101,226,198,95,38,50,195,136,18,48,92,110,179,107,44,218,31,180,2,93,99,126,153,13,147,191,222,14,184,94,113,208,202,145,42,216,200,17,3,250,64,116,49,139,54,102,107,245,160,220,42,3,174,241,127,227,106,59,27,232,124,63,234,198,73,219,39,103,32,80,203,212,187,216,135,7,88,204,146,185,52,102,75,143,158,111,85,225,128,246,111,214,227,186,6,23,157,61,216,104,121,56,157,152,252,148,120,6,218,34,133,63,205,126,102,72,52,182,84,103,211,159,2,253,93,210,61,134,169,0,103,150,101,37,75,172,168,243,10,112,167,191,21,98,245,33,215,81,53,235,30,67,167,141,84,42,210,93,119,159,1,122,104,241,22,57,68,47,169,193,103,90,190,61,44,224,189,21,88,147,53,135,169,45,174,218,18,7,130,19,216,35,126,11,145,228,73,153,245,219,39,150,1,96,56,164,143,13,61,164,101,250,228,175,250,132,19,149,251,14,175,255,81,200,250,20,23,163,38,250,207,245,230,166,201,23,175,171,61,59,145,67,50,80,189,177,56,20},
  {97,245,128,229,171,89,93,105,173,56,255,68,131,89,98,225,219,32,11,248,241,30,158,141,135,32,77,114,120,161,208,216,142,45,44,95,47,62,115,143,74,114,153,94,16,117,234,128,221,134,117,193,253,193,45,215,240,151,212,120,213,175,185,141,157,63,240,151,106,102,220,117,115,114,121,222,3,25,6,179,167,240,217,150,86,212,153,234,218,237,206,241,41,145,191,129,217,52,186,164,231,132,106,210,133,42,82,202,167,51,72,105,169,207,243,18,221,205,185,30,199,27,110,80,126,238,114,121,233,81,52,164,111,27,20,222,113,175,136,60,136,195,253,22,178,29,237,199,149,123,67,39,109,62,68,5,172,230,92,110,95,87,191,173,53,128,222,180,61,231,234,247,135,248,19,168,133,219,252,247,38,134,176,202,4,29,65,128,181,47,88,214,157,45,133,197,197,203,126,118,34,167,89,174,187,29,255,170,241,59,132,120,85,140,187,49,171,31,134,20,47,161,76,192,30,171,130,241,10,176,237,216,159,23,221,14,227,210,123,25,133,55,210,103,255,247,90,169,113,30,176,10,244,64,27,38,134,16,175,45,111,174,221,151,234,125,244,127,231,93,101,112,78,63,60,47,17,34,248,76,237,43,226,2,228,46,101,177,93,7,15,63,198,184,30,41,42,146,76,0,145,189,31,37,192,226,73,2,34,83,249,179,37,215,15,91,237,152,249,146,119,111,48,55,142,204,173,236,47,197,146,100,149,62,192,81,64,203,191,152,111,87,96,218,48,32,83,48,199,185,152,40,61,68,173,178,104,22,62,55,54,73,251,187,213,75,101,49,236,13,62,96,10,33,101,235,139,62,254,17,51,249,224,94,80,2,16,170,229,96,19,175,69,52,67,177,217,178,82,3,0,157,140,13,104,35,76,196,68,140,43,244,7,169,184,215,238,212,98,1,183,101,102,96,225,225,115,108,93,72,214,100,238,211,31,65,74,88,47,177,187,162,62,53,86,153,228,24,115,58,45,156,4,178,84,194,97,170,216,214,175,95,26,178,228,79,1,236,121,179,238,196,146,191,41,36,85,248,100,202,94,20,137,164,146,19,23,137,254,32,72,228,32,154,72,129,195,199,238,188,82,112,196,47,131,76,52,253,67,205,226,62,159,182,9,190,119,135,190,58,112,156,119,72,244,91,49,248,217,128,13,175,189,142,53,254,135,102,179,18,112,160,175,159,248,162,25,116,168,235,26,199,138,206,214,126,28,225,247,53,64,186,202,27,2,97,159,18,32,96,114,228,62,73,61,161,245,171,106,165,168,140,99,234,252,170,193,183,200,32,192,120,168,128,59,145,14,136,101,173,21,215,56,114,50,225,72,232,165,206,48,235,214,70,117,3,14,157,92,62,174,158,220,236,107,83,4,211,244,239,243,227,96,131,217,117,33,243,64,237,185,74,132,201,225,247,145,249,180,224,230,118,122,127,0,143,89,76,217,160,127,49,127,207,119,145,147,240,158,166,103,205,134,226,71,154,202,192,194,72,179,119,177,128,78,51,104,92,128,19,48,162,66,103,137,17,39,25,209,150,142,146,70,132,81,24,75,148,248,106,92,217,16,236,82,136,120,138,32,124,219,245,143,38,46,72,11,6,156,135,82,121,224,215,206,64,200,81,86,77,153,53,31,144,100,232,25,6,187,160,133,104,81,245,237,224,187,135,89,110,42,103,25,56,39,48,4,162,154,216,92,106,184,203,205,143,63,206,133,80,28,79,71,96,246,128,56,122,97,1,23,10,224,132,37,78,72,25,166,102,208,171,13,59,117,187,110,215,164,80,204,67,112,38,144,145,13,157,254,93,105,91,135,221,9,175,111,236,161,82,207,37,92,100,85,254,154,151,129,169,215,242,166,42,51,215,71,137,236,166,14,231,2,75,93,198,231,219,199,108,31,119,98,63,35},
  {11,241,176,168,3,187,54,170,104,67,126,159,134,53,132,12,124,226,152,192,56,104,116,63,207,228,46,239,63,231,73,172,183,91,99,199,237,224,15,41,164,51,106,95,139,179,132,228,4,5,150,29,37,114,178,169,124,233,178,241,14,189,62,216,174,47,97,157,58,125,241,88,242,109,56,93,20,116,84,217,160,85,212,15,19,201,34,21,132,211,160,26,119,18,147,9,15,175,207,68,156,181,78,255,30,37,126,3,223,221,254,1,89,8,92,96,4,18,25,181,15,93,85,214,155,224,255,226,41,253,188,80,94,94,15,14,55,80,71,46,83,152,136,211,69,233,185,197,96,218,39,76,87,178,121,17,29,218,179,17,81,160,20,185,227,121,2,91,191,86,7,141,109,5,174,127,34,111,168,29,214,16,244,98,8,246,96,24,31,174,199,13,41,195,194,175,53,225,44,6,103,154,133,169,90,182,183,52,209,171,46,156,12,83,150,222,207,209,130,60,185,69,18,250,15,38,38,49,222,94,200,60,159,71,54,114,239,222,71,255,229,181,13,217,167,161,10,87,195,113,79,78,225,235,58,48,195,206,159,215,243,62,180,128,18,218,113,198,91,120,150,96,176,123,59,174,215,202,181,124,194,44,231,227,15,13,246,89,217,236,236,38,195,247,125,238,17,246,143,52,42,163,221,91,207,87,125,115,147,93,203,235,197,126,34,187,98,3,93,72,68,116,2,228,118,158,85,78,70,243,123,82,50,199,96,180,37,45,16,126,74,99,87,202,68,48,63,123,127,223,33,150,184,133,221,61,105,113,181,217,102,176,150,16,136,101,237,191,7,213,171,244,250,167,12,196,29,189,3,211,122,212,11,219,26,124,223,33,249,230,53,124,99,135,123,203,20,188,155,15,125,93,200,21,96,181,6,93,42,161,17,186,138,102,111,47,68,73,83,120,66,45,252,159,46,39,135,154,146,164,164,104,100,191,5,171,47,95,53,1,217,51,50,229,180,47,148,46,139,52,183,0,199,43,189,63,34,152,15,161,8,224,187,243,230,150,132,32,122,91,24,141,248,85,234,65,2,78,133,204,173,21,85,92,17,252,188,169,0,104,58,153,167,91,212,209,120,210,161,247,47,47,164,249,130,140,113,146,255,193,104,11,135,127,97,5,6,169,161,213,75,234,113,245,234,230,105,201,117,115,169,120,36,48,126,136,192,30,4,187,135,201,2,165,125,18,144,101,164,230,27,136,219,62,213,134,81,234,229,100,36,19,216,93,203,184,65,175,129,45,245,249,164,132,214,202,66,26,251,94,110,218,76,0,78,173,201,28,4,203,155,91,184,94,36,192,161,239,98,125,108,22,219,105,31,11,110,152,145,115,86,189,82,34,191,86,13,197,12,219,128,24,64,100,236,116,221,143,51,168,245,106,199,121,187,218,244,119,19,255,189,92,150,243,161,191,190,120,71,119,61,72,205,171,129,235,217,245,231,115,146,157,93,220,49,220,242,16,53,180,171,231,172,175,80,127,202,140,83,122,125,22,48,146,226,20,40,75,230,78,248,190,237,3,45,170,154,126,76,34,230,22,126,3,140,141,67,150,189,75,137,80,253,221,11,15,79,16,175,105,212,154,168,63,139,194,59,254,130,198,203,99,89,84,98,108,74,95,123,31,186,12,30,200,54,146,128,131,205,171,56,195,229,128,211,131,95,204,218,204,74,226,104,106,90,99,69,168,143,156,16,120,1,130,71,167,71,56,97,5,86,108,164,206,61,183,30,159,242,18,254,129,30,238,222,227,136,90,15,14,117,115,72,147,145,49,234,73,100,93,7,94,100,204,197,23,91,229,15,240,71,6,7,121,144,87,204,115,104,14,69,139,211,12,84,220,11,196,97,67,242,68,162,112,57,22,231,111,111,96,93,91,94,78,128,24,123,75,87,134,65,40,245,251,15},
  {153,191,212,239,198,31,55,51,203,108,237,223,29,194,159,88,237,223,205,6,202,5,205,17,132,65,182,179,96,139,112,227,255,191,84,143,191,29,90,168,89,200,165,74,65,211,60,181,84,100,202,57,80,108,58,35,140,192,169,39,15,23,119,32,202,182,44,191,89,199,121,1,22,248,6,236,252,84,48,244,240,172,94,61,94,233,236,172,196,9,248,94,25,230,44,89,91,24,117,215,166,157,250,117,3,119,152,162,113,34,52,127,15,61,105,116,43,230,129,135,43,239,85,186,23,70,254,140,244,209,229,69,90,80,79,157,226,34,55,5,27,3,227,60,158,248,180,96,177,60,203,138,3,206,97,119,111,111,180,238,251,49,142,62,72,76,209,20,170,177,84,173,161,124,242,182,199,96,176,235,136,162,77,10,166,184,187,190,77,33,92,243,179,152,112,241,44,25,206,16,74,70,162,224,80,210,38,226,198,241,212,122,8,58,170,196,115,43,228,37,71,9,196,102,14,122,106,70,241,85,146,227,125,111,57,38,236,127,244,142,56,136,209,6,232,155,24,223,89,162,141,163,202,18,129,126,129,186,3,121,20,218,184,222,216,128,113,96,23,67,69,165,138,255,43,56,90,121,138,245,210,45,231,227,162,171,110,89,222,117,255,94,103,54,53,193,234,112,116,251,173,182,25,28,123,123,177,66,56,54,202,68,112,12,240,186,255,221,234,238,177,209,105,209,6,28,81,233,73,202,236,224,242,47,53,87,25,177,237,4,69,143,200,1,189,175,108,197,191,161,185,9,4,223,91,160,2,212,189,16,122,0,66,155,187,88,92,128,225,41,78,40,204,106,113,209,108,106,12,101,212,108,208,37,245,27,68,60,148,125,150,173,104,12,12,254,184,151,165,237,252,177,150,120,188,119,17,20,245,133,137,44,187,236,103,90,249,251,122,60,94,201,3,31,216,161,224,181,9,39,139,161,195,183,5,207,188,178,26,205,88,160,41,74,38,39,78,117,74,66,150,105,161,147,87,108,179,223,123,100,64,208,74,59,99,40,252,46,170,222,149,186,39,43,100,239,102,104,114,237,141,20,172,108,254,135,99,133,181,78,77,234,68,104,201,26,68,104,93,48,9,92,65,94,108,164,36,230,51,7,11,210,22,200,150,100,76,230,15,239,150,225,65,151,123,121,120,164,142,102,21,26,106,248,193,154,19,166,208,155,214,226,166,210,244,81,252,183,248,68,16,240,228,97,115,179,81,89,226,114,175,2,201,88,229,167,82,118,137,34,139,236,183,162,172,53,108,108,159,221,75,103,129,85,78,54,21,70,236,247,125,178,109,79,120,4,22,25,197,135,52,127,198,208,159,136,151,19,116,68,115,83,215,182,1,82,212,244,178,8,73,172,229,105,162,248,191,100,102,225,102,198,247,122,130,179,33,237,127,108,230,246,39,182,193,102,102,70,119,39,125,140,80,125,39,252,123,172,137,129,65,243,96,254,132,116,180,239,203,199,116,79,117,11,30,243,171,144,111,107,68,62,121,127,83,108,237,178,209,148,132,30,98,155,222,239,82,148,122,194,19,81,228,8,160,122,40,108,148,10,3,74,159,115,247,127,169,95,93,50,222,177,177,148,155,147,86,200,25,122,107,232,167,87,133,232,113,138,89,72,185,148,89,207,140,135,167,66,35,84,139,21,161,254,96,59,186,228,191,8,63,55,96,147,156,210,78,122,230,161,216,42,194,76,165,85,219,207,233,160,243,203,45,172,35,253,241,66,136,71,57,199,236,246,218,106,1,112,205,99,230,201,183,103,118,1,38,6,181,68,253,49,3,25,205,212,89,112,184,175,1,156,12,41,23,32,30,122,140,27,9,237,37,57,49,36,30,232,209,160,28,124,127,76,129,48,252,170,98,69,192,158,82,44,177,252,208,61,231,97,12,74,179,28,1},
} ;

static const int16_t precomputed_encode_653x1541_x[precomputed_encode_653x1541_NUM][crypto_encode_ITEMS] = {
  {-23786,14210,30180,8105,13150,21276,-8289,11107,-25022,31483,-5443,11880,-69,31327,-13439,1985,-25591,-19825,23156,-3868,-14611,-22669,-28577,18515,10802,17330,-22299,-15520,-17076,31670,4627,10185,22332,-28054,31736,8993,-6555,7353,-1581,23738,25027,-19523,1248,-15696,3914,15935,-1539,15507,21410,-596,28130,14042,-26386,-3000,-24687,-31734,32459,27443,-21387,-244,-36,-31838,-1381,-26223,24582,29725,-9607,16695,25127,-28083,6694,32033,-3232,-31972,15159,-6034,894,17644,-23077,-31688,-26695,-25880,-19288,11256,-7589,-17301,-24332,-30143,11081,32460,18137,-6692,-8984,29440,12404,-18488,9115,14050,-1483,29493,18373,5005,-8639,2746,25268,5236,25441,25569,-3050,5262,-10788,1738,-3268,-13498,-27224,16622,29664,14869,-2025,-6285,-137,20012,13888,22309,16548,31842,-4257,3436,16607,828,-21108,10573,-29171,-14867,32217,14065,28054,-9631,-25425,12023,28904,-24015,27997,29116,-2238,13549,-25527,-16172,11909,-10275,-31526,15827,-29514,15729,30234,7734,2100,-17032,1326,-7047,-19785,10232,8397,-17311,4997,-3197,-6551,-27944,-10210,-29893,11593,11464,17545,-30555,-15521,21342,-4740,19935,27331,31140,-28898,-27782,31919,18565,-1384,15107,4523,-2054,4791,-31462,-30197,5663,-5166,13965,-1799,-19818,-29825,31250,19435,9114,18668,-30621,2556,6373,9478,8499,-673,1848,-10492,-30843,25394,-14370,397,24754,-1433,-21865,-18421,8679,-12331,-25750,15780,20430,-22826,-2673,21846,-26739,-8908,-26062,12978,27048,2229,-9449,-31631,28718,14866,10235,-24076,11236,22437,1513,-19306,6457,-25165,24453,-12731,-2942,7463,28471,-964,-3599,-30713,-5381,22679,-29131,-22337,-31364,-11282,-21639,-32040,-12595,-20860,-9574,4037,-11707,-7244,-23292,-30627,519,2937,-16271,-15416,19137,11316,-13388,-7757,-4325,32681,5916,18992,-15770,-13652,14287,-28795,-2349,2412,26382,18945,31428,-23258,-3626,-21419,-20292,-5467,30131,-10986,-12937,-14162,30150,3962,-29140,-32567,-3111,-5140,7097,10253,3054,22489,32230,-27896,-705,-7978,11149,9659,-23365,-1738,-32185,16155,5338,-6119,29477,25979,12819,30651,23035,-28688,4563,8677,-12971,-28930,-9685,20806,26784,-6715,-5406,18337,-32662,4671,-19597,-5014,3071,29812,22837,32069,27458,-25683,-12160,6962,-7603,-1735,-21258,-19779,5405,10875,-10568,-31790,-17332,-15068,5288,5773,26616,-22705,-22260,-31909,-10196,22623,-12649,13140,15145,7077,31871,-3535,-11516,29977,-16927,29032,-31680,-12040,-15411,1225,1087,-15674,21940,-721,30979,-32518,-20402,12601,-13727,-4701,-29281,12928,-22282,-5643,-31718,-6571,23938,-15258,-30648,17158,-18835,-310,4913,-16153,-1800,23973,11458,2523,-965,12889,-26142,-6392,14752,28637,-31488,-10612,16076,-214,-6592,21568,26990,-13614,-15561,10633,27946,-22829,-26595,29583,6186,8063,-26293,16549,-31445,24925,26479,22488,32087,-18161,-14577,29038,-22117,3314,-7863,20344,27006,-14224,-348,-16466,18003,-30956,1556,-32190,-6060,-16260,13793,26115,-11175,-21086,-21620,31270,-14246,-27707,-11289,-30279,3866,8323,-6214,-13543,31746,17611,3535,-9197,26935,31192,32354,-19777,-16491,13939,11411,-9987,139,14353,-12501,20558,28062,-10689,24312,-19484,23280,-1496,9993,-21526,-25000,-10037,11904,-18378,-17941,-25087,22634,6778,24718,4245,-12794,23044,-26669,-2637,-16341,10630,-2453,23042,11883,8857,-17485,-23299,641,326,-10921,19100,32449,30142,-839,27918,-8085,24531,-27764,18436,5740,32260,17104,-26927,-4124,18593,-28730,433,32623,-21558,29129,-2543,18761,8486,2141,8809,495,10152,11621,-4234,8664,-3573,-6575,91,-16020,32338,17629,-24737,10610,-9730,-21697,-1589,-16737,18960,-3368,-27387,-31114,-8605,20053,15194,-5551,12415,24403,-11955,-19226,10604,-18081,999,6107,9829,31481,623,19906,-3838,-28071,-30267,6409,10753,29554,10876,30501,17819,-16748,25363,-24240,-16384,12366,-17794,28849,27192,9437,16810,-4630,23400,10570,-18023,25356,27448,14225,27201,-12816,530,-13806,-19530,10384,1524,-11171,18818,-12899,23715,-3141,21696,24992,3009,-14469,21989,1077,-20785,-10877,-13290,2587,-15031,-27811,3371,19162,-8037,-1162,-20578,-15088,-12415,-10836,-12224,17248,-29552,7517,26100,-32215,-6368,-16101,13094,-20947,27270,22190,-27955,12717,2062,-16175,23256},
  {6720,-9997,-27559,-30136,-15722,20010,-22401,-5578,-10394,27110,26667,22822,5126,14179,14819,23667,-5020,-29261,18386,-13224,-9074,32681,-12155,419,-15791,-9824,-6128,-30779,32697,13122,26235,2766,11441,-2009,18945,27099,-27154,29394,-30901,-4940,16028,16134,17909,4316,-1384,9122,5587,-20290,14118,-29014,5702,-23911,10489,-32367,4600,5441,-7574,15917,26838,28975,27299,-8924,-31524,-22141,-28586,27760,-18348,-31280,17113,-8677,11773,-16999,-27501,-29191,-947,-18878,-23240,19610,-24463,-16122,15393,29216,-11981,27086,-30925,11699,3519,-23285,26691,16785,-16133,17627,-5909,-11585,-13986,24883,-1961,1593,18313,-15773,24791,5603,3168,22005,9462,-7686,-7671,-1940,28082,-28788,26812,-4830,-17761,-21909,6208,16764,-5399,1151,-16293,15970,15376,12303,15898,14020,-26045,11510,-287,-3961,28936,-8207,538,-12159,-30549,-3480,15006,5122,-12056,-22364,-18419,-7321,308,-11864,12025,-23128,-1153,602,21722,15685,32626,22317,27169,18962,29179,4196,-1019,6664,-17804,31053,-5124,4147,-23432,-14779,21378,-14787,1387,-28775,227,-16805,-30064,-25276,20049,-22474,16099,-29707,25398,176,16190,1547,11308,-30438,15050,8219,10190,2249,-21711,14345,20931,-11529,7232,15968,-12730,11781,5154,-9149,-15113,-17687,11366,-9488,-21486,9817,18127,-8612,-29889,30576,7859,14870,-7998,13343,19533,1440,29016,-25292,-30685,-2062,1188,-7136,-19784,11013,-24128,16370,21372,3954,8451,-12795,23245,25859,28090,-27772,31779,-10055,3311,26786,12274,29724,-9080,5723,-5595,12233,9882,12012,-14235,29737,-20884,13147,9708,3188,27283,-12378,8036,26214,-10738,6840,16604,27307,31221,30841,-25,13963,-11091,-24568,-25182,7418,-24432,115,-10089,16567,2979,13699,-20940,-22992,7122,-30379,-8001,27518,-25937,-20366,-1679,9013,-27217,764,29722,17938,10078,-983,19596,-25737,21036,29286,-3028,-4208,-26407,16353,21973,4854,-4692,10062,29953,-5806,-16418,-4481,29951,15547,7338,25533,21068,16120,433,-6609,-31786,-25865,-16241,-21434,-29022,-28350,-13789,-29776,20602,3200,-2030,31699,25614,18891,18974,-28172,31526,32664,22131,4938,4398,17065,18295,611,-24322,18661,29038,30082,-18888,23161,-19006,24208,-23159,26270,-29495,8856,-24204,-13259,-22966,16257,-12978,12072,-12865,21459,-4506,9678,31443,-13535,-1260,18318,-14536,-1611,14396,-22215,-13096,-32737,-25642,-20116,-15741,8282,10092,-12724,-743,-24160,-22074,6497,-32184,18334,-27456,-7973,23115,-22112,-2747,25070,31899,-19722,-3150,1434,-9645,-2659,13295,11239,31762,-6239,-10136,21697,-18851,26464,-14190,-31293,-12387,-3910,10051,-18687,-22238,27427,-28524,7186,-25632,12650,-29012,-7507,-8485,1657,-26542,-31255,30651,9120,4001,18642,19181,-7697,1324,18650,4217,-6871,-32578,32007,-15438,195,-28015,26138,-32000,19173,-16383,-32518,-32038,-12830,7324,-3335,-17364,-17718,-14469,-19613,-23247,13393,30244,22253,6160,-31127,27373,-30347,-20622,26953,2587,10153,23327,-22499,10996,11931,-23291,-260,31624,-7877,-24251,-19755,-26813,-26498,-23828,-27023,9272,-14886,3897,-32469,-22944,1083,-32593,-12460,13531,32280,25725,-26948,5472,17361,29066,7321,14701,2913,16654,17000,31402,-8899,6049,-11923,-5635,-13877,-32004,-11526,24848,17824,-124,-22747,-19296,-11913,-19560,-9962,1461,23143,-15315,-10574,-13091,28331,-13995,7219,22068,27149,17073,-20614,12206,-28198,-6166,14823,25654,-21351,-25761,-841,-25378,1567,14467,-20474,26840,-31192,-19578,11716,-7010,14188,-9136,-5323,-23302,25528,-14093,-31334,24541,6102,16077,-32337,13277,18992,21805,-8066,-17249,19351,20678,-24234,9760,27186,-22636,-16092,25150,-21938,-11694,-5149,1190,13107,24369,7638,-13353,-5186,14340,-27294,27606,19494,-6198,6586,-27851,-30625,-22920,3045,-28560,-20349,22108,-27311,28589,11385,-2024,-10726,7620,20964,25463,8886,31884,29937,-905,45,-14468,14177,22005,-31660,22060,28805,16734,-3710,26006,14387,-10855,-24397,-5644,16706,6180,-1220,347,-17881,22503,-4423,22816,-788,5568,572,-8054,31730,954,14240,17653,-18644,26593,28421,2387,-3789,27643,-29521,20662,11129,-11589,-17448,-12366,14795,8524,17222,27031,6001,-18103,13175,24855,1597,-8885,-1439,28794,-18074,-30364,-20474,7457,-24197,22211,30952,-19077},
  {-146,30800,23953,26374,-10242,3941,-18263,-10230,32606,-9445,1848,26259,4127,28335,9123,24331,19572,21946,24053,32189,-12191,-9779,-5558,23679,2861,-1880,-16002,-5528,9189,233,-23375,14660,-18972,23063,21524,-10523,23365,8980,26391,5019,26667,12256,27141,20125,-7153,-24107,-3849,24199,-25653,-8320,10985,-7699,30710,-31568,20939,2453,19088,14338,-25227,14856,-12066,31634,24163,-32230,-14715,-15845,20289,2647,23105,8386,-25480,1266,-23268,17078,-15584,23110,31903,-11907,-7761,9045,29317,-1879,-18446,5507,31450,27611,6734,28297,-1559,-31918,-29029,-10272,5300,-27458,17373,29066,-24633,-4345,32036,-22947,21741,-7080,4275,-32383,13511,24910,-28136,-7724,22495,-20681,-31421,28804,-813,5145,29435,3515,11473,16956,589,-32441,-7889,-11323,21056,-23612,6608,-9688,-10625,1277,18528,-5655,-31806,5206,7797,-14974,-12361,18124,8072,-29132,4044,14875,2884,-899,28975,17138,-23763,-11747,25298,15275,-564,5210,-14907,-22875,-5696,-14830,29808,-20029,-19108,29210,15015,8538,3227,-26103,-4432,-657,-23068,14002,-25612,-28512,28230,6237,28161,25050,2569,-31884,17873,6927,84,29957,-28996,-174,21873,-22870,-18989,1708,-3084,-31370,29111,12974,-22501,-8065,2317,-11472,19487,28207,19328,-1564,17183,-14736,27797,32518,2645,24697,27898,-4145,32428,27244,9173,27600,-3188,-21572,8745,22331,-16700,3205,-16898,22935,-16053,-19035,-9078,15972,-7043,24032,25395,31574,12879,6072,24154,12116,13439,28183,14543,19414,-29252,10045,-32492,5480,-822,-2693,-4802,22771,30778,21086,1071,20213,-14190,-29628,-15932,26860,1440,11626,-25469,-13721,11003,-10912,-28811,-27285,-26945,27832,26458,-25883,27371,11119,-12848,11622,9865,-1000,5784,-953,-20228,-21143,7289,25989,1680,24061,8174,-8608,18308,9944,-31288,-15277,32347,30030,-24952,3129,9302,-15710,-28660,-19645,-30325,-15761,28506,-909,-19009,-20363,-20838,18103,10243,-24781,-16917,-12927,9604,30711,-10920,-11754,23935,4220,14934,9608,-11992,20947,-20387,15756,-20548,-28904,-1623,-32626,-27452,19262,-11508,-3532,-30387,10164,-28646,7100,6580,13282,29552,8377,25956,-3576,-20767,24956,1442,11228,-1956,15902,13414,1573,23770,9782,16870,8655,-2721,11939,20166,-27628,30253,-15320,-30362,5762,-26684,-19631,12644,-14751,-15611,1903,23239,-18837,32216,32735,-11337,-26743,-12847,6975,6130,-2725,17785,20919,-4459,-26859,-25134,7830,15424,19854,-32153,-12198,-12400,-19114,-6225,32217,14876,3487,-28309,11398,-29446,32582,-6161,-10022,30998,-30306,26206,-22961,-15347,-4430,-7708,-14303,-26245,31791,-3535,5723,31749,-24213,-8270,14387,24911,22585,-20465,20778,-15295,23893,-15255,-16354,-24766,-10815,1392,11113,23119,-16382,-32138,5499,-22732,26072,1270,-9792,10915,-24505,2266,19662,-520,23178,-20347,-15670,-19762,-6579,-7602,-20582,26904,11485,27946,-18512,17916,14955,-2810,21692,-28925,-18683,8720,2700,14994,-12570,-31137,24061,-23802,-31934,-3907,-15478,-26390,-4324,14573,-6908,-3208,20620,-11580,-27095,20578,26539,10148,9987,29861,-1900,14185,4343,-9449,-26492,19509,22184,-22996,7760,-6116,-26380,-20963,-30697,8566,-8625,22171,23369,9025,29756,11324,9195,8923,-12702,32420,13280,1595,12937,8855,2642,31605,-13632,-31957,6484,-6823,-13493,10047,9024,-14967,-14383,-26341,32433,18339,14480,2240,-12421,10355,-16494,-16830,-14056,3414,20433,7977,6686,32609,19826,20120,-9810,-31448,-7515,7023,30344,-32111,-23097,-10511,19264,15156,-4660,-17859,191,-25772,1585,-9722,26964,4780,4318,3292,27805,-26029,11007,-32485,29719,31799,-28636,-15711,-6257,-15928,26698,-20905,25634,12032,16812,25622,-31893,15679,-17299,-21369,10909,-21958,-802,7810,21738,3096,13359,-29429,-26535,8499,-2266,2132,29137,-1541,-784,27535,29342,4222,-2211,19496,21862,-31073,572,14899,-29022,-11543,30709,27068,-10684,1335,-28131,21565,-17956,-18932,15486,-26816,-6958,21729,2096,-20773,-25977,-9655,-6315,-30134,-7116,-11027,31742,21334,-614,-2047,-32448,-28677,21485,-24298,13256,-29631,-25752,4285,28717,12193,-28335,30184,-9705,-5464,26195,24865,18942,-20915,12273,28234,-12767,-29061,-17229,26556,18994,-7388,15694,14524,-14260,7577,3855,-17626,15227,-15376,-5001,15866,-7590,-27826},
  {-15405,-3437,-13617,-9456,-30823,-3135,-5650,-26943,20620,-31046,28048,-7338,3025,2494,8559,31694,3980,88,-26135,-17652,14913,19732,-2378,-30527,8284,-8968,24166,-24691,10338,10924,-25977,30997,15062,-2057,22402,-26164,18294,-23356,2052,-414,28207,-3119,-17618,22004,-19571,12223,19180,15368,-14212,-9218,-25002,-13295,-27968,-11864,5439,5310,28578,-15700,1994,-11022,29679,-29835,4971,4050,-16311,-22498,8824,-22596,-18499,-31071,23286,17234,-5809,-17468,19219,-6496,10652,19216,-2473,9754,-6981,-19703,24895,4403,8177,-21714,-15830,-12310,-24402,32664,-14787,-25595,-27813,-10741,-7972,-11051,-10338,-15552,15655,30510,15496,-31092,-20888,24021,-21635,-29116,26519,-7294,6759,26120,11136,-25149,11149,23141,12030,24839,9337,26133,27497,-31594,-2024,9932,-18262,-8629,-22713,25038,3139,-15310,11923,-5752,-27153,-5730,11325,31752,-27206,-1003,11974,14652,9785,-11468,-29449,25310,2517,-27133,-18844,1985,7606,17156,-5511,14208,-13558,23326,-24662,4206,-4145,-1329,20155,16574,-5180,-20737,-81,-3943,1912,-17131,32681,-21333,4415,-1313,-30870,19127,-12080,-23819,-26974,31284,-10856,-12477,-25496,-13119,-5704,-18061,710,24066,29494,7104,73,-8557,-14991,-29689,-16043,29547,12806,25543,26818,5489,-6946,-5360,-21915,22265,-15987,-14189,28735,18307,8775,16253,-8459,13613,-29704,-27018,9334,-9807,-6398,-21490,12693,-17917,4984,-4848,5505,-26280,29610,16211,16196,1139,1581,-22722,-24594,57,-30804,5013,-25783,-31035,23173,27993,29440,27771,31613,-22882,7869,-20292,-5135,11802,-23966,27391,6368,-15215,-27911,27898,22602,3932,8332,-26593,28390,-6616,-8580,-23585,-16810,-27998,-10449,2178,-30927,-22879,-25159,27127,17995,-19230,23835,3165,-551,-32155,22328,10219,-12353,8190,-3299,27260,-23120,-21012,-8579,4303,29043,-25933,23310,2158,-31590,-16471,21660,15152,15650,-14364,25123,21943,28987,27829,-7048,4224,-17538,1205,-23512,-29912,21296,25262,28764,-4152,31334,12806,3762,13529,-25093,-11811,6875,6511,-1158,18525,20141,29977,1860,1377,-20233,11419,-9495,-21720,-24949,-7511,18026,-31536,-4661,-31655,-18766,-31389,-19750,13407,20654,-9672,3591,-18774,-3498,-5593,9775,-1859,-8164,-28937,13252,22994,-8271,19160,3128,-8778,3495,-19881,25503,-23087,-25736,-10162,-11884,-16709,-12477,-6669,27305,11124,-10979,-1252,-28428,11804,-13871,7577,-20830,2874,-23278,-12905,-1166,7547,-27787,15186,31686,30377,-25026,23699,9970,-25044,-7128,23609,2990,-30859,24514,-12647,29913,30215,20839,23259,-10853,30391,17891,-19413,-19972,30691,-26348,5176,10632,-1241,9393,-13343,-12279,-27254,-26062,1305,21606,-16569,3620,-31981,-15690,21357,6656,12613,-29457,-12883,-5274,17211,31939,-5045,3931,-7663,-18960,10722,-11222,-21444,-28110,-12555,32170,31228,8597,26588,-29507,-2639,-17006,-14047,-18909,12300,-10637,-23187,22172,28912,-3213,-2699,14051,-23905,-41,-9046,15669,-21664,22827,-9222,18502,31030,31738,8252,-28107,12927,18791,-4746,4981,-5507,30962,-17631,-1129,30256,23050,-29577,-30610,-5235,10616,-14177,-25385,3915,10696,5061,31821,107,-28474,-11113,23841,9340,-19438,-15335,27099,20767,4005,-23005,15971,11294,28310,-2252,-22322,-6251,-17520,-12461,25294,26863,-13701,-3509,9853,26639,28827,22882,-31909,-13391,-24222,6548,-31753,12396,1214,4745,22020,25926,29378,-7545,2992,-8736,29423,-17460,-11348,8891,-29171,25807,22340,27452,22665,3611,-20541,-25102,-13667,29941,-3044,-6694,21840,6958,7663,-22729,-26248,14626,24838,-23213,11091,-15495,-4667,-31638,32063,-17088,909,-32035,-17559,-12751,-19678,-23991,-1450,-31969,-10922,-4968,4115,-13291,-22533,8540,811,-12280,13510,-9849,-78,19603,31665,2194,-10866,-20109,-10223,18936,17462,3071,22680,-29986,-25401,11710,31400,-28744,12747,-10954,2190,24138,32578,4703,14564,-13535,-20882,-7061,19938,-31372,-12192,1601,-29451,29606,14077,20287,1968,4665,-18209,-1840,-6762,-4074,1555,-14382,-23458,27896,-17792,17348,13625,-26209,23715,24696,-24048,-13178,-30425,-27527,5013,-15977,12265,-20053,3,30499,-30957,929,25444,-8211,-7954,-474,-28862,-10454,1753,-7218,-23804,-9755,19132,11719,-3723,22283,9963,9014,30659,-23778,-30419,-23965,-24265,20487,-19408,-7691,13895,-4560,-18204},
} ;

static void test_encode_653x1541_impl(long long impl)
{
  unsigned char *s = test_encode_653x1541_s;
  unsigned char *x = test_encode_653x1541_x;
  unsigned char *s2 = test_encode_653x1541_s2;
  unsigned char *x2 = test_encode_653x1541_x2;
  long long slen = crypto_encode_STRBYTES;
  long long xwords = crypto_encode_ITEMS;
  long long xlen;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_encode_653x1541_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_encode = ntruprime_dispatch_encode_653x1541(impl);
    printf("encode_653x1541 %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_encode_653x1541_implementation(impl),ntruprime_dispatch_encode_653x1541_compiler(impl));
  } else {
    crypto_encode = ntruprime_encode_653x1541;
    printf("encode_653x1541 selected implementation %s compiler %s\n",ntruprime_encode_653x1541_implementation(),ntruprime_encode_653x1541_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 1024;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {
      xlen = xwords*crypto_encode_ITEMBYTES;

      output_prepare(s2,s,slen);
      input_prepare(x2,x,xlen);
      endianness(x,xwords,crypto_encode_ITEMBYTES);
      secret(x,xlen);
      crypto_encode(s,x);
      public(x,xlen);
      endianness(x,xwords,crypto_encode_ITEMBYTES);
      public(s,slen);
      checksum(s,slen);
      output_compare(s2,s,slen,"crypto_encode");
      input_compare(x2,x,xlen,"crypto_encode");

      double_canary(s2,s,slen);
      double_canary(x2,x,xlen);
      endianness(x2,xwords,crypto_encode_ITEMBYTES);
      secret(x2,xlen);
      crypto_encode(s2,x2);
      public(x2,xlen);
      endianness(x2,xwords,crypto_encode_ITEMBYTES);
      public(s2,slen);
      if (memcmp(s2,s,slen) != 0) fail("failure: crypto_encode is nondeterministic\n");
    }
    checksum_expected(encode_653x1541_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_encode_653x1541_NUM;++precomp) {
    output_prepare(s2,s,crypto_encode_STRBYTES);
    input_prepare(x2,x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    memcpy(x,precomputed_encode_653x1541_x[precomp],crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    memcpy(x2,precomputed_encode_653x1541_x[precomp],crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    crypto_encode(s,x);
    if (memcmp(s,precomputed_encode_653x1541_s[precomp],crypto_encode_STRBYTES)) {
      fail("failure: crypto_encode fails precomputed test vectors\n");
      printf("expected s: ");
      for (long long pos = 0;pos < crypto_encode_STRBYTES;++pos) printf("%02x",((unsigned char *) precomputed_encode_653x1541_s[precomp])[pos]);
      printf("\n");
      printf("received s: ");
      for (long long pos = 0;pos < crypto_encode_STRBYTES;++pos) printf("%02x",s[pos]);
      printf("\n");
    }
    output_compare(s2,s,crypto_encode_STRBYTES,"crypto_encode");
    input_compare(x2,x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES,"crypto_encode");
  }
}

void test_encode_653x1541(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"encode")) return;
  if (targetp && strcmp(targetp,"653x1541")) return;
  storage_encode_653x1541_s = callocplus(crypto_encode_STRBYTES);
  test_encode_653x1541_s = aligned(storage_encode_653x1541_s,crypto_encode_STRBYTES);
  if (crypto_encode_STRBYTES > maxalloc) maxalloc = crypto_encode_STRBYTES;
  storage_encode_653x1541_x = callocplus(crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
  test_encode_653x1541_x = aligned(storage_encode_653x1541_x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
  if (crypto_encode_ITEMS*crypto_encode_ITEMBYTES > maxalloc) maxalloc = crypto_encode_ITEMS*crypto_encode_ITEMBYTES;
  storage_encode_653x1541_s2 = callocplus(maxalloc);
  test_encode_653x1541_s2 = aligned(storage_encode_653x1541_s2,crypto_encode_STRBYTES);
  storage_encode_653x1541_x2 = callocplus(maxalloc);
  test_encode_653x1541_x2 = aligned(storage_encode_653x1541_x2,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);

  for (long long offset = 0;offset < 1;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("encode_653x1541 offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_encode_653x1541();++impl)
      forked(test_encode_653x1541_impl,impl);
    ++test_encode_653x1541_s;
    ++test_encode_653x1541_x;
    ++test_encode_653x1541_s2;
    ++test_encode_653x1541_x2;
  }
  free(storage_encode_653x1541_x2);
  free(storage_encode_653x1541_s2);
  free(storage_encode_653x1541_x);
  free(storage_encode_653x1541_s);
}
#undef crypto_encode_STRBYTES
#undef crypto_encode_ITEMS
#undef crypto_encode_ITEMBYTES

