///////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) Crossbar.io Technologies GmbH and contributors and contributors.
//
// Boost Software License - Version 1.0 - August 17th, 2003
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
//
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//
///////////////////////////////////////////////////////////////////////////////

#include "exceptions.hpp"
#include "wamp_message.hpp"
#include "wamp_transport_handler.hpp"

#include <boost/asio/buffer.hpp>
#include <boost/asio/placeholders.hpp>
#include <boost/asio/read.hpp>
#include <boost/asio/write.hpp>
#include <system_error>

namespace autobahn {

inline wamp_websocket_transport::wamp_websocket_transport(
    const std::string& uri,
    bool debug_enabled)
    : wamp_transport()
    , m_connect()
    , m_disconnect()
    , m_message_unpacker()
    , m_debug_enabled(debug_enabled)
    , m_uri(uri)
{
}

inline boost::future<void> wamp_websocket_transport::connect()
{
    if (is_open()) {
        m_connect.set_exception(boost::copy_exception(network_error("network transport already connected")));
        return m_connect.get_future();
    }

    async_connect(m_uri, m_connect);

    return m_connect.get_future();
}

inline boost::future<void> wamp_websocket_transport::disconnect()
{
    if (!is_open()) {
        throw network_error("network transport already disconnected");
    }

    close();

    m_disconnect.set_value();
    return m_disconnect.get_future();
}

inline bool wamp_websocket_transport::is_connected() const
{
    return is_open();
}

inline void wamp_websocket_transport::send_message(wamp_message&& message)
{
    auto buffer = std::make_shared<msgpack::sbuffer>();
    msgpack::packer<msgpack::sbuffer> packer(*buffer);
    packer.pack(message.fields());

   
    // Write actual serialized message.
    write(buffer->data(), buffer->size());

    if (m_debug_enabled) {
        std::cerr << "TX message (" << buffer->size() << " octets) ..." << std::endl;
        std::cerr << "TX message: " << message << std::endl;
    }
}

inline void wamp_websocket_transport::set_pause_handler(pause_handler&& handler)
{
    m_pause_handler = std::move(handler);
}

inline void wamp_websocket_transport::set_resume_handler(resume_handler&& handler)
{
    m_resume_handler = std::move(handler);
}

inline void wamp_websocket_transport::pause()
{
    if (m_pause_handler) {
        m_pause_handler();
    }
}

inline void wamp_websocket_transport::resume()
{
    if (m_resume_handler) {
        m_resume_handler();
    }
}

inline void wamp_websocket_transport::attach(
    const std::shared_ptr<wamp_transport_handler>& handler)
{
    if (m_handler) {
        throw std::logic_error("handler already attached");
    }

    m_handler = handler;

    m_handler->on_attach(this->shared_from_this());
}

inline void wamp_websocket_transport::detach()
{
    if (!m_handler) {
        throw std::logic_error("no handler attached");
    }

    m_handler->on_detach(true, "wamp.error.goodbye");
    m_handler.reset();
}

inline bool wamp_websocket_transport::has_handler() const
{
    return m_handler != nullptr;
}


inline void wamp_websocket_transport::receive_message(const std::string& msg)
{
    if (m_debug_enabled) {
        std::cerr << "RX message received." << std::endl;
    }

    if (m_handler) {
        m_message_unpacker.reserve_buffer(msg.size());
        memcpy(m_message_unpacker.buffer(), msg.data(), msg.size());
        m_message_unpacker.buffer_consumed(msg.size());

        msgpack::unpacked result;

        while (m_message_unpacker.next(&result)) {
            wamp_message::message_fields fields;
            result.get().convert(fields);

            wamp_message message(std::move(fields), std::move(*(result.zone())));
            if (m_debug_enabled) {
                std::cerr << "RX message: " << message << std::endl;
            }

            m_handler->on_message(std::move(message));
        }
    }
    else {
        std::cerr << "RX message ignored: no handler attached" << std::endl;
    }
}

} //namespace autobahn