// Copyright 1999-2020 - Universit de Strasbourg/CNRS
// The Aladin Desktop program is developped by the Centre de Donnes
// astronomiques de Strasbourgs (CDS).
// The Aladin Desktop program is distributed under the terms
// of the GNU General Public License version 3.
//
//This file is part of Aladin Desktop.
//
//    Aladin Desktop is free software: you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Aladin Desktop is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    The GNU General Public License is available in COPYING file
//    along with Aladin Desktop.
//

package cds.aladin;

import java.util.Enumeration;
import java.util.HashMap;

import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeModel;

/**
 * Gestion d'un model associ au Directory Tree
 * @author Pierre Fernique [CDS]
 * @version 1.0 Janvier 2017 - cration
 */
public class DirectoryModel extends DefaultTreeModel {
   protected DefaultMutableTreeNode root;
   private Aladin aladin;
   
   protected DirectoryModel(Aladin aladin) {
      super( new DefaultMutableTreeNode( new TreeObj(aladin,"root",null,Directory.ROOT_LABEL,"") ) );
      root = (DefaultMutableTreeNode) getRoot();
      this.aladin = aladin;
   }
   
   /** Colorations (noire, verte ou orange) des branches de l'arbre en fonction de l'tat des feuilles */
   protected int populateFlagIn() { return populateFlagIn(root); }
   private int populateFlagIn(DefaultMutableTreeNode node) {
      TreeObj treeObj = (TreeObj) node.getUserObject();
      if( treeObj.isHidden && node.isLeaf() ) return -2;
      if( node.isLeaf() ) return treeObj.getIsIn();

      DefaultMutableTreeNode subNode = null;
      Enumeration e = node.children();
      int rep = -2;
// Mthode de colaration initiale => le vert n'tait pas contaminant
//      while( e.hasMoreElements() ) {
//         subNode = (DefaultMutableTreeNode) e.nextElement();
//         int isIn =  populateFlagIn(subNode);
//         if( rep==-2 ) rep=isIn;
//         else if( rep!=isIn ) rep=-1;
//      }
      while( e.hasMoreElements() ) {
         subNode = (DefaultMutableTreeNode) e.nextElement();
         int isIn =  populateFlagIn(subNode);
         if( isIn==-2 ) continue;
         if( rep==-2 ) rep=isIn;
         else if( isIn==1 ) rep=isIn;             // Le vert est contaminant
         else if( isIn==-1 && rep==0 ) rep=isIn;  // Le blanc aussi mais uniquement si orange prcdemment
      }

      treeObj.setIn(rep);
      return rep;
   }
   
   /** Comptage de la descendance de chaque branche (nombre de noeuds terminaux d'une branche)
    * Mmorisation dans TreeObj, soit en tant que rfrence (hs!=null), soit
    * en tant que dcompte courant
    * @param hs mmorisation des valeurs sous forme path=n (ex: /Image/Optical=32, ...)
    */
   protected int countDescendance() { return countDescendance(null); }
   protected int countDescendance(HashMap<String,Integer> hs) {
      if( root.isLeaf() ) return 0;
      int nb=countDescendance(root.toString(),root,hs);
      return nb;
  }
   private int countDescendance(String prefix,DefaultMutableTreeNode parent,HashMap<String,Integer> hs) {
      TreeObj to = (TreeObj) parent.getUserObject();
      if( parent.isLeaf() )  return 1;

      int n=0;
      Enumeration e = parent.children();
      while( e.hasMoreElements() ) {
         DefaultMutableTreeNode node = (DefaultMutableTreeNode) e.nextElement();
         n += countDescendance( prefix, node,hs );
      }
      
      // Mmorisation de rfrence
      if( hs!=null ) {
         hs.put(to.path, n ); 
         to.nb = to.nbRef = n;
         
      //Dcompte temporaire
      } else {
         to.nb = n;
     }
      
      return n;
   }

   // Mmorisation des noeuds parents dans une hashmap pour les retrouver rapidement
   private HashMap<String, DefaultMutableTreeNode> fastAccess=null;
   
   /** Rinitialisation des lments acclrateurs de la cration de l'arbre */
   protected void resetCreate() {
      fastAccess = new HashMap<>();
   }
   
   // Mmorisation d'un noeud par son path afin de le retrouver rapidement
   private void memoFast( DefaultMutableTreeNode node ) {
      TreeObj to = (TreeObj) node.getUserObject();
      fastAccess.put( to.path, node );
   }
   
   // Retourne le noeud correspondant au path, null sinon
   private DefaultMutableTreeNode findFast( String path ) {
      return fastAccess.get( path );
   }
   
   /** Insertion d'un noeud, ventuellement avec les diffrents lments de sa branche
    * si ceux-ci n'existent pas encore. Conserve le noeud parent de l'insertion
    * dans lastParentNode afin d'acclerer une ventuelle insertion ultrieure au mme endroit
    * @param treeObj Le nouveau noeud  insrer
    */
   protected void createTreeBranch(TreeObj treeObj) {
      // Cration immdiate car c'est le mme parent que la prcdente insertion
      if( createWithExistingParent( treeObj) ) return;
      
      // Cration rcursive (ventuellement pour la branche)
      DefaultMutableTreeNode nodeUp [] = new DefaultMutableTreeNode[1];
      int index [] = new int[1];
      DefaultMutableTreeNode lastParentNode = createTreeBranch( this, root, treeObj, 0, nodeUp, index);
      if( lastParentNode!=null ) memoFast(lastParentNode);
   }
   
   /** Mthode interne - Tentative d'insertion d'un noeud sur le parent de la dernire insertion. Retourne true
    * si l'insertion est effectivement possible, false sinon */
   private boolean createWithExistingParent( TreeObj treeObj) {
      int pos = lastSlash(treeObj.path);
      if( pos==-1 ) return true;
      String path = treeObj.path.substring(0, pos);
      
      DefaultMutableTreeNode node = findFast( path );
      if( node==null ) return false;
      node.add( new DefaultMutableTreeNode(treeObj) );
      return true;
   }
   
   private int lastSlash( String s ) {
      return s.lastIndexOf('/');
      
      // BIZARRE, CA NE MARCHE PAS  !!
//      for( int i=s.length()-1; i>=0; i-- ) {
//         if( s.charAt(i)=='/' ) {
//            if( i>0 && s.charAt(i-1)=='\\' ) continue;
//            return i;
//         }
//      }
//      return -1;
   }

   /** Mthode interne - Insertion rcursive d'un noeud en fonction du "path" du noeud  insrer.
    * Cration ventuelle des noeuds des branches si ceux-ci n'existent pas encore
    * @param model Le modle associ  l'arbre
    * @param parent Le noeud courant du parcours de l'arbre (root au dbut)
    * @param treeObj Le noeud  insrer
    * @param opos L'index courant dans le "path" du noeud, -1 si le path a t compltement parcouru
    *             (ex: Optical/Image/DSS2/color ) => pos = index du I de Image
    * @param parentUp tableau  (1er lement) servant  mmoriser le noeud parent de l'insertion de la branche
    *               la plus haute dans l'arbre (pour pouvoir avertir les listeners de la greffe de la branche)
    * @param childIndex tableau  (1er lement) servant  mmoriser l'indice de la brance greffe au plus haut
    *              dans l'arbre (pour pouvoir avertir les listeners de la greffe de la branche)
    * @return Le parent direct de l'insertion du noeud (afin de pouvoir insrer plus rapidement un autre noeud au mme endroit)
    */
   private DefaultMutableTreeNode createTreeBranch(DefaultTreeModel model, DefaultMutableTreeNode parent, 
         TreeObj treeObj, int opos, DefaultMutableTreeNode parentUp [], int childIndex []) {

      // Dtermination du prochain lment dans le path
      // Rq: On dcoupe par "/" mais sans prendre en compte "\/"
      int pos, offset=opos;
      do  {
         pos=treeObj.path.indexOf('/',offset);
         offset=pos;
         if( pos>1 && treeObj.path.charAt(pos-1)=='\\') offset++;
         else offset=-1;
      } while( offset!=-1 );

      // Dtermination du label courant et de son path
      String label = pos<0 ? treeObj.path.substring(opos) : treeObj.path.substring(opos,pos);
      String path = pos<0 ? treeObj.path : treeObj.path.substring(0,pos);
      
      // Les noeuds utilisateurs n'utilisent pas de checkbox pour cet arbre ( supprimer si possible)
      ((TreeObj)parent.getUserObject()).noCheckbox();

      try {
         // Recherche du fils qui correspond  l'emplacement o la greffe doit avoir lieu
         DefaultMutableTreeNode subNode = null;
         Enumeration e = parent.children();
         while( e.hasMoreElements() ) {
            subNode = (DefaultMutableTreeNode) e.nextElement();
            TreeObj fils = (TreeObj) subNode.getUserObject();
            if( label.equals(fils.label) ) break;
            subNode=null;
         }

         // Aucun fils ne correspond, il faut donc crer la branche (ou insrer le noeud terminal si on est au bout)
         if( subNode==null ) {
            
            // Noeud terminal ? c'est donc celui  insrer
            if( pos==-1 ) {
               subNode = new DefaultMutableTreeNode( treeObj );
            }
            
            // Branche intermdiaire ? dj connue ou non ?
            else  subNode = new DefaultMutableTreeNode( new TreeObj(aladin,"",null,label,path) );
            
            // Cas tordu o le pre tait en fait une branche terminal  tord
            // Je greffe alors l'objet terminal associ en tant que fils, et je rectifie la nature
            // de l'objet associ au pre.
            TreeObj pere = (TreeObj) parent.getUserObject();
            if( pere!=null && pere instanceof TreeObjDir) {
               if( Aladin.levelTrace>=3 ) System.err.println("Directory tree clash on "+pere.path+" (supposed to be a leaf, and in fact a node)");
               DefaultMutableTreeNode pereNode = new DefaultMutableTreeNode( pere );
               parent.add( pereNode );
               parent.setUserObject( new TreeObj(aladin,"",null,pere.label,pere.path)  );
            }
            parent.add(subNode);
         }
         
         // On n'est pas au bout du path, il faut donc continuer rcursivement
         // (en fait, une boucle serait plus adapte, mais comme on ne descend jamais
         // bien profond, a ne va pas gner
         if( pos!=-1 ) return createTreeBranch(model, subNode, treeObj, pos + 1, parentUp, childIndex);
         
         // Retourne le noeud parent
         return parent;

      } catch( Exception e ) {
         e.printStackTrace();
      }
      return null;
   }
}

